package lu.uni.minus.ui;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.SwingWorker.StateValue;
import javax.swing.border.TitledBorder;
import javax.swing.text.BadLocationException;
import javax.swing.text.html.HTMLDocument;
import javax.swing.text.html.HTMLEditorKit;

import lu.uni.minus.preferences.DataSet;
import lu.uni.minus.utils.TextPaneWorker;
import lu.uni.minus.utils.sp.SPDetectWorker;

public class DetectSPPanel extends JPanel implements Announcer
{
  private static final long serialVersionUID = -6186413526615829959L;
  private final MainWindow mw;
  private final Dimension space = new Dimension(6, 6);
  private JTextField tfDistance;
  private JTextField tfTime;
  private JTextField tfThreshold;
  private JButton btnStart;
  private JButton btnCancel;
  private Action cancelAction;
  private JTextPane messageArea;
  private JProgressBar progressBar;
  private TextPaneWorker worker;
  private HTMLEditorKit kit;
  private HTMLDocument doc;
  private DataSet dataset;

  public DetectSPPanel(MainWindow aMw, DataSet aDataset) {
    mw = aMw;
    dataset = aDataset;
    createGui();
  }

  private void createGui() {
    final JSplitPane splitPane = new JSplitPane();
    setLayout(new BorderLayout(0, 0));
    add(splitPane, BorderLayout.CENTER);
    final JPanel mainPanel = new JPanel();
    mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.Y_AXIS));
    JPanel titlePanel = new JPanel();
    titlePanel.setLayout(new BoxLayout(titlePanel, BoxLayout.X_AXIS));
    titlePanel.setBorder(new TitledBorder(null, "Parameters for detecting stay points of Geolife",
        TitledBorder.LEADING, TitledBorder.TOP, null, null));
    JPanel leftPanel = new JPanel();
    leftPanel.setLayout(new BoxLayout(leftPanel, BoxLayout.Y_AXIS));

    leftPanel.setMaximumSize(new Dimension(250, 100));
    leftPanel.add(createLabelPanel("Distance threshold :"));
    leftPanel.add(Box.createRigidArea(space));
    leftPanel.add(createLabelPanel("Time threshold :"));
    leftPanel.add(Box.createRigidArea(space));
    leftPanel.add(createLabelPanel("Stay point merging threshold :"));
    leftPanel.add(Box.createRigidArea(space));
    titlePanel.add(leftPanel);

    JPanel rightPanel = new JPanel();
    rightPanel.setLayout(new BoxLayout(rightPanel, BoxLayout.Y_AXIS));
    Dimension fieldSize = new Dimension(100, 20);
    tfDistance = new JTextField("30");
    tfDistance.setPreferredSize(fieldSize);
    tfDistance.setMaximumSize(fieldSize);
    tfDistance
        .setToolTipText("<html>\r\nIt is the minimum time length that a continuous set of GPS points lasts to be a stay\r\n<br/>\r\n point. A typical value is 30.\r\n<html/>");
    tfTime = new JTextField("100");
    tfTime.setPreferredSize(fieldSize);
    tfTime.setMaximumSize(fieldSize);
    tfTime
        .setToolTipText("<html>\r\nIt is the maximum distance between the first point and the last point of a \r\n<br/>\r\ncontinuous set of GPS points for them to be a stay point. Typical values are \r\n<br/>\r\nintegers like 100, 150 and 200.\r\n</html>");
    tfThreshold = new JTextField("30");
    tfThreshold.setPreferredSize(fieldSize);
    tfThreshold.setMaximumSize(fieldSize);
    tfThreshold
        .setToolTipText("<html>\r\nIt is the maximum distance for two preliminary stay points to be merged into one.\r\n<br/>\r\nA common practice is to make it take the same value as the distance threshold.\r\n<html/>");
    rightPanel.add(createComponentPanel(tfDistance, " m"));
    rightPanel.add(createComponentPanel(tfTime, " s"));
    rightPanel.add(createComponentPanel(tfThreshold, " m"));

    titlePanel.add(rightPanel);
    titlePanel.add(Box.createHorizontalGlue());
    mainPanel.add(titlePanel);

    JScrollPane scrollMessage = new JScrollPane();
    messageArea = new JTextPane();
    messageArea.setPreferredSize(new Dimension(300, 200));
    messageArea.setContentType("text/html");
    messageArea.setEditable(false);
    // messageArea.setBackground(Color.BLACK);
    kit = new HTMLEditorKit();
    messageArea.setEditorKit(kit);
    doc = new HTMLDocument();
    messageArea.setDocument(doc);
    scrollMessage.setViewportView(messageArea);
    mainPanel.add(scrollMessage);

    splitPane.setRightComponent(mainPanel);

    final JPanel usersPanel = new JPanel();
    usersPanel.setBorder(null);
    usersPanel.setLayout(new BorderLayout(0, 0));

    JScrollPane scrollPane = new JScrollPane();
    usersPanel.add(new JLabel("User list:"), BorderLayout.NORTH);
    usersPanel.add(scrollPane, BorderLayout.CENTER);
    final JList<String> usersList = new JList<String>();
    usersList.setListData(dataset.getUserList());

    scrollPane.setViewportView(usersList);
    usersPanel.setPreferredSize(new Dimension(100, 500));
    splitPane.setLeftComponent(usersPanel);

    JPanel buttonPanel = new JPanel();
    buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
    buttonPanel.add(Box.createRigidArea(space));
    progressBar = new JProgressBar(0,100);
    progressBar.setStringPainted(true);
    progressBar.setVisible(false);
    progressBar.setPreferredSize(new Dimension(800,20));
    buttonPanel.add(Box.createHorizontalGlue());
    buttonPanel.add(progressBar);
    buttonPanel.add(Box.createHorizontalGlue());
    btnStart = new JButton("Start");
    final DetectSPPanel me = this;
    btnStart.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent e) {
        messageArea.setText("");
        List<String> selectedList = usersList.getSelectedValuesList();
        ArrayList<String> filteredList;
        if (selectedList.size() == 0) {
          JOptionPane.showMessageDialog(null, "Please first choose some users.", "Choose users",
              JOptionPane.INFORMATION_MESSAGE);
          return;
        }
        double distanceThreshold = Double.parseDouble(tfDistance.getText().trim());
        double timeThreshold = Double.parseDouble(tfTime.getText().trim());
        double mergingThreshold = Double.parseDouble(tfThreshold.getText().trim());
        if (distanceThreshold <= 0.0 || timeThreshold <= 0.0 || mergingThreshold <= 0.0) {
          JOptionPane.showMessageDialog(null, "The thresholds must be positive numbers.",
              "Number format error", JOptionPane.ERROR_MESSAGE);
          return;
        }
        File spDir = dataset.getSPDir(tfDistance.getText(), tfTime.getText(), tfThreshold.getText());
        if (spDir.exists()) {
          filteredList = new ArrayList<String>();
          HashSet<String> doneUsers = new HashSet<String>(Arrays.asList(spDir.list()));
          if (doneUsers.size() > 0) {
            for (String user : selectedList) {
              if (!doneUsers.contains(user+".txt")) {
                filteredList.add(user);
              }
            }
          }
        }
        else {
          filteredList = new ArrayList<String>(selectedList);
        }
        if (filteredList.size() > 0) {
          btnStart.setEnabled(false);
          btnCancel.setEnabled(true);
          progressBar.setVisible(true);
          worker = new SPDetectWorker(dataset, filteredList, tfDistance.getText().trim(), tfTime
              .getText().trim(), tfThreshold.getText().trim());
          worker.setAnnouncer(me);
          worker.addPropertyChangeListener(new PropertyChangeListener()
          {
            public void propertyChange(final PropertyChangeEvent event) {
              if (event.getPropertyName().equals("progress")) {
                progressBar.setIndeterminate(false);
                progressBar.setValue((Integer) event.getNewValue());
              }
              else {
                if (event.getPropertyName().equals("state")) {
                  switch ((StateValue) event.getNewValue()) {
                    case DONE:
                      btnStart.setEnabled(true);
                      btnCancel.setEnabled(false);
                      // workDone();
                      break;
                    case STARTED:
                    case PENDING:
                      progressBar.setIndeterminate(true);
                      break;
                  }
                }
              }
            }
          });
          worker.execute();
        }
        else{
          addMessage("All selected users have staypoints generated - nothing to be done");
          progressBar.setVisible(false);
        }
      }
    });
    buttonPanel.add(btnStart);
    buttonPanel.add(Box.createRigidArea(space));

    cancelAction = new AbstractAction("Cancel")
    {
      private static final long serialVersionUID = 7666652133260327969L;

      public void actionPerformed(final ActionEvent e) {
//         mw.goHome();
        // cancel();
        worker.cancel(true);
      }
    };
    btnCancel=new JButton(cancelAction);
    btnCancel.setEnabled(false);
    buttonPanel.add(btnCancel);
    buttonPanel.add(Box.createRigidArea(space));

    JButton btnClose = new JButton("Close");
    btnClose.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent e) {
        mw.goHome();
      }
    });
    buttonPanel.add(btnClose);
    mainPanel.add(buttonPanel);
  }

  private JPanel createLabelPanel(String text) {
    JPanel result = new JPanel();
    result.setLayout(new BoxLayout(result, BoxLayout.X_AXIS));
    result.add(Box.createHorizontalGlue());
    result.add(Box.createRigidArea(space));
    result.add(new JLabel(text));
    result.add(Box.createRigidArea(space));
    return result;
  }

  private JPanel createComponentPanel(JComponent component, String text) {
    JPanel result = new JPanel();
    result.setLayout(new BoxLayout(result, BoxLayout.X_AXIS));
    result.add(Box.createRigidArea(space));
    result.add(component);
    result.add(new JLabel(text));
    result.add(Box.createHorizontalGlue());
    return result;
  }

  public void addMessage(String html) {
    try {
      kit.insertHTML(doc, doc.getLength(), html, 0, 0, null);
    }
    catch (BadLocationException e) {
      e.printStackTrace();
    }
    catch (IOException e) {
      e.printStackTrace();
    }
  }
}
