/**
 * Author: Piotr Kordy (piotr.kordy@uni.lu <mailto:piotr.kordy@uni.lu>)
 * Date:   06/06/2014
 * Copyright (c) 2014,2013 University of Luxembourg
 * All rights reserved.
 * Licensed under GNU Affero General Public License 3.0;
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU Affero General Public License as
 *    published by the Free Software Foundation, either version 3 of the
 *    License, or (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Affero General Public License for more details.
 *
 *    You should have received a copy of the GNU Affero General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * Author: Piotr Kordy (piotr.kordy@uni.lu <mailto:piotr.kordy@uni.lu>)
 */
package lu.uni.minus.ui;

import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.io.IOException;

import javax.swing.AbstractAction;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

import lu.uni.minus.preferences.DataSet;
import lu.uni.minus.ui.epub.EpubViewer;
import lu.uni.minus.utils.ProcessRunner;

import org.apache.commons.io.FileUtils;

public class MainMenu extends JMenuBar
{
  private static final long serialVersionUID = -6736607039942971939L;
  private static ADAction fileAdd;
  private static ADAction fileEdit;
  private static ADAction fileDelete;
  private static ADAction fileViewData;
  private static ADAction fileViewUser;
  private static ADAction fileReset;
  private static ADAction fileExit;

  private static ADAction profileSP;
  private static ADAction profileRoI;
  private static ADAction profileFPS;
  private static ADAction profileVisualize;

  private static ADAction similarityCU;
  private static ADAction similarityMSF;

  private static ADAction helpManual;
  private static ADAction helpStartPage;
  private static ADAction helpAbout;
  private static MainWindow mw;
  private static JPanel manual = null;
  private static DataFlowPane background = null;

  public MainMenu(MainWindow aMw) {
    super();
    mw = aMw;
    background = new DataFlowPane();
    createActions();
    helpStartPage.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_PERFORMED, null));
    this.add(createFileMenu());
    this.add(createUserProfileMenu());
    this.add(createUserSimilMenu());
    this.add(createHelpMenu());
  }

  /**
   * Create all actions used in application.
   */
  private void createActions()
  {
    fileAdd = new ADAction("Add Dataset")
    {
			private static final long serialVersionUID = -618696261748903055L;

			public void actionPerformed(final ActionEvent e)
      {
        AddDSPane addPane=new AddDSPane(mw);
        mw.setContent(addPane);
      }
    };
    fileAdd.setMnemonic(KeyEvent.VK_A);
    fileAdd.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A,
        InputEvent.ALT_MASK));
//     addDS.setSmallIcon(new ImageIcon(this.getClass().getResource(
//         "/icons/new.png")));
    fileAdd.setToolTip("Add new DataSet");

    fileEdit = new ADAction("Edit Dataset")
    {
			private static final long serialVersionUID = -7289253008537115308L;

			public void actionPerformed(final ActionEvent e)
      {
        DefaultMutableTreeNode node = getNode();
		    if (node != null ) {
			    EditDSPanel panel = new EditDSPanel(mw, node);
          mw.setContent(panel);
        }
      }
    };
    fileEdit.setMnemonic(KeyEvent.VK_E);
    fileEdit.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E,
        InputEvent.ALT_MASK));
    fileEdit.setToolTip("Edit DataSet output path");

    fileDelete = new ADAction("Delete Dataset")
    {
			private static final long serialVersionUID = 6047147350467024375L;

			public void actionPerformed(final ActionEvent e)
      {
        DefaultMutableTreeNode node = getNode();
		    if (node != null ) {
					DataSet dataset = (DataSet) node.getUserObject();
			    int ans = JOptionPane.showConfirmDialog(mw,
           "Delete also output directory:\""+dataset.getOutputPath().toString()+"\" ?", "Confirm",
					JOptionPane.YES_NO_CANCEL_OPTION);

			    if (ans != JOptionPane.CANCEL_OPTION) {
            DefaultTreeModel treeModel = (DefaultTreeModel) (mw.getDataSetPane().getTree().getModel());
				    treeModel.removeNodeFromParent(node);
            mw.getSettings().removeDataSet(dataset);

			      if (ans == JOptionPane.YES_OPTION) {
				      try {
				      	FileUtils.deleteDirectory(dataset.getOutputPath());
				      } catch (IOException ex) {
				      	JOptionPane.showMessageDialog(mw,
                  "An error deleting \""+dataset.getOutputPath()+"\":"+ex.getMessage(),
				      	  "Error", JOptionPane.ERROR_MESSAGE);
				      	return;
				      }
            }
          }
        }
      }
    };
    fileDelete.setMnemonic(KeyEvent.VK_D);
    fileDelete.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D,
        InputEvent.ALT_MASK));
    fileDelete.setToolTip("Delete DataSet");

    fileViewData = new ADAction("View Dataset's Info")
    {
			private static final long serialVersionUID = -5761368747300826480L;

			public void actionPerformed(final ActionEvent e)
      {
        DataSet ds =getDataSet();
		    if (ds != null ) {
			    StatDSPanel panel = new StatDSPanel(mw,ds);
          mw.setContent(panel);
        }
      }
    };
    fileViewData.setMnemonic(KeyEvent.VK_V);
    fileViewData.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_I,
        InputEvent.ALT_MASK));
    fileViewData.setToolTip("View Dataset's Info");

    fileViewUser = new ADAction("View User Stats")
    {
			private static final long serialVersionUID = 9195585245323216127L;

			public void actionPerformed(final ActionEvent e)
      {
        DataSet ds =getDataSet();
		    if (ds != null ) {
			    ViewUserPanel panel = new ViewUserPanel(mw,ds);
          mw.setContent(panel);
        }
      }
    };
    fileViewUser.setMnemonic(KeyEvent.VK_U);
    fileViewUser.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S,
        InputEvent.ALT_MASK));
    fileViewUser.setToolTip("View User Stats");

    fileReset = new ADAction("Reset Settings")
    {
			private static final long serialVersionUID = -1991063302215411335L;

			public void actionPerformed(final ActionEvent e)
      {
        mw.getSettings().clearPreferences();
        mw.getSettings().setDefault();
      }
    };
    fileReset.setMnemonic(KeyEvent.VK_R);
    fileReset.setToolTip("Reset the settings of the program to the default values");

    fileExit = new ADAction("Exit")
    {
      private static final long serialVersionUID = -6586817922511469697L;

      public void actionPerformed(final ActionEvent e)
      {
        WindowEvent windowClosing = new WindowEvent(mw,
            WindowEvent.WINDOW_CLOSING);
        mw.dispatchEvent(windowClosing);
      }
    };
    fileExit.setMnemonic(KeyEvent.VK_X);
    fileExit.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q,
        InputEvent.CTRL_MASK));
    fileExit.setSmallIcon(new ImageIcon(getClass().getResource(
        "/icons/exit.png")));

    profileSP = new ADAction("Detect stay points")
    {
      private static final long serialVersionUID = -8654654165123569697L;

			public void actionPerformed(final ActionEvent e)
      {
        DataSet ds =getDataSet();
		    if (ds != null ) {
          if(ds.getType()==DataSet.Type.GPS){
			      DetectSPPanel panel = new DetectSPPanel(mw,ds);
            mw.setContent(panel);
          }
          else{
		 	     JOptionPane.showMessageDialog(mw,
		 	     		"We need to detect stay points only for GPS type of dataset.",
		 	     		"Wrong dataset", JOptionPane.INFORMATION_MESSAGE);
          }
        }
        
      }
    };
    profileSP.setMnemonic(KeyEvent.VK_S);

    profileRoI = new ADAction("Generate RoI")
    {
      private static final long serialVersionUID = -8122226658463666667L;
			public void actionPerformed(final ActionEvent e)
      {
        DataSet ds =getDataSet();
		    if (ds != null ) {
			    GenerateRoIPanel panel = new GenerateRoIPanel(mw,ds);
          mw.setContent(panel);
        }
      }
    };
    profileRoI.setMnemonic(KeyEvent.VK_G);

    profileFPS = new ADAction("Extract frequent pattern sets")
    {
      private static final long serialVersionUID = -1566545631658665457L;
			public void actionPerformed(final ActionEvent e)
      {
        DataSet ds =getDataSet();
		    if (ds != null ) {
			    ExtractFPSPanel panel = new ExtractFPSPanel(mw,ds);
          mw.setContent(panel);
        }
      }
    };
    profileFPS.setMnemonic(KeyEvent.VK_E);

    profileVisualize = new ADAction("Visualize")
    {
			private static final long serialVersionUID = -2350956653269249550L;
			public void actionPerformed(final ActionEvent e)
      {
        if(!ProcessRunner.isWindows()){
		 	     JOptionPane.showMessageDialog(mw,
		 	     		"The tool for visualisation works only under Microsoft Windos.",
		 	     		"Wrong system", JOptionPane.INFORMATION_MESSAGE);
           return;
        }
        DataSet ds =getDataSet();
		    if (ds != null ) {
			    VisualisePanel panel = new VisualisePanel(mw,ds);
          mw.setContent(panel);
        }
      }
    };
    profileVisualize.setMnemonic(KeyEvent.VK_V);

    similarityCU = new ADAction("Compare users")
    {
      private static final long serialVersionUID = 9095216058966524914L;

      public void actionPerformed(final ActionEvent e)
      {
        DataSet ds =getDataSet();
		    if (ds != null ) {
			    CompareUsersPanel panel = new CompareUsersPanel(ds);
          mw.setContent(panel);
        }
      }
    };
    similarityCU .setMnemonic(KeyEvent.VK_C);

    similarityMSF = new ADAction("Manage semantic files")
    {
      private static final long serialVersionUID = 5439405409660547113L;

      public void actionPerformed(final ActionEvent e)
      {
        DataSet ds =getDataSet();
		    if (ds != null ) {
			    ManageDistFilesPanel panel = new ManageDistFilesPanel(ds);
          mw.setContent(panel);
        }
      }
    };
    similarityMSF .setMnemonic(KeyEvent.VK_M);

    helpManual = new ADAction("Help contents")
    {
			private static final long serialVersionUID = 1354242457785397409L;
			public void actionPerformed(final ActionEvent e)
      {
        if (manual==null){
          manual = new EpubViewer();
        }
        mw.setContent(manual);
      }
    };
    helpManual.setMnemonic(KeyEvent.VK_H);
    helpManual.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F1,0));
    helpManual.setSmallIcon(new ImageIcon(getClass().getResource("/icons/help.png")));

    helpStartPage = new ADAction("Start page")
    {
			private static final long serialVersionUID = 3364237774054136963L;

			public void actionPerformed(final ActionEvent e)
      {
        final JScrollPane scrollPane = new JScrollPane(background);
        scrollPane.setAutoscrolls(true);
        mw.setContent(scrollPane);
      }
    };
    helpStartPage.setMnemonic(KeyEvent.VK_S);
    helpStartPage.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_H,
        InputEvent.CTRL_MASK));
    helpStartPage.setSmallIcon(new ImageIcon(getClass().getResource("/icons/home.png")));

    helpAbout = new ADAction("About")
    {
			private static final long serialVersionUID = 6650789708603730200L;

			public void actionPerformed(final ActionEvent e)
      {
        AboutDialog ad = new AboutDialog(mw);
        ad.setVisible(true);
      }
    };
    helpAbout.setMnemonic(KeyEvent.VK_A);
    helpAbout.setSmallIcon(new ImageIcon(getClass().getResource("/icons/about.png")));
  }

  private final JMenu createFileMenu() {
    JMenu menu = new JMenu("File");
    menu.setMnemonic('F');
    menu.add(fileAdd);
    menu.add(fileEdit);
    menu.add(fileDelete);
    menu.addSeparator();
    menu.add(fileViewData);
    menu.add(fileViewUser);
    menu.addSeparator();
    menu.add(fileReset);
    menu.add(fileExit);
    return menu;
  }

  private JMenu createUserProfileMenu() {
    JMenu menu = new JMenu("User Profile");
    menu.setMnemonic('P');
    menu.add(profileSP);
    menu.add(profileRoI);
    menu.add(profileFPS);
    menu.addSeparator();
    menu.add(profileVisualize);
    return menu;
  }

  private JMenu createUserSimilMenu() {
    JMenu menu = new JMenu("User Similarity");
    menu.setMnemonic('U');
    menu.add(similarityCU);
    menu.add(similarityMSF);
    return menu;
  }

  private JMenu createHelpMenu() {
    JMenu menu = new JMenu("Help");
    menu.setMnemonic(KeyEvent.VK_H);
    menu.add(helpManual);
    menu.add(helpStartPage);
    menu.add(helpAbout);
    return menu;
  }

  /**
   * @return the helpStartPage
   */
  public static ADAction getHelpStartPage() {
    return helpStartPage;
  }

  /**
   * ADAction represents an action that is used in application.
   */
  public abstract class ADAction extends AbstractAction
  {
    /**
     *
     */
    private static final long serialVersionUID = 8109471079193338016L;

    /**
     * Defines an ADAction object with the specified descripiton and a default
     * icon.
     *
     * @param text
     *          text to be displayed
     */
    public ADAction(final String text) {
      super(text);
    }

    /**
     * Sets accelerator for the action.
     *
     * @param accelerator
     *          new accelerator
     */
    public final void setAccelerator(final KeyStroke accelerator) {
      putValue(ACCELERATOR_KEY, accelerator);
    }

    /**
     * Sets the new small icon for the action.
     *
     * @param icon
     *          new icon
     */
    public final void setSmallIcon(final Icon icon) {
      putValue(SMALL_ICON, icon);
    }

    /**
     * Sets tooltip for the action.
     *
     * @param text
     *          new tooltip text
     */
    public final void setToolTip(final String text) {
      putValue(SHORT_DESCRIPTION, text);
    }

    /**
     * Sets long description for the action.
     *
     * @param text
     */
    public final void setDescription(final String text) {
      putValue(LONG_DESCRIPTION, text);
    }

    /**
     * Sets mnemonic for the action.
     *
     * @param mnemonic
     *          new mnemonic
     */
    public final void setMnemonic(final Integer mnemonic) {
      putValue(MNEMONIC_KEY, mnemonic);
    }
  }

  private DataSet getDataSet(){
    DefaultMutableTreeNode node= getNode();
    if(node!=null){
      return (DataSet)node.getUserObject();
    }
    return null;
  }
  
  private DefaultMutableTreeNode getNode(){
     DefaultMutableTreeNode node = mw.getDataSetPane().getSelected();
		 if (node == null ) {
		 	JOptionPane.showMessageDialog(mw,
		 			"Please first choose a dataset by selecting a node.",
		 			"Choose a dataset", JOptionPane.INFORMATION_MESSAGE);
      return null;
		 }
     else{
       return node;
     }
  }
}
