package lu.uni.minus.utils;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import lu.uni.minus.preferences.DataSet;
import lu.uni.minus.ui.AddDSPane;

/**
 * Checks the validity of datasets
 *
 * @author Piotr Kordy
 * @author
 * @author
 * @version
 */
public class ValidityWorker extends TextPaneWorker
{
  /** The dataset to be processed */
  private final DataSet dataset;
  /**
   * Parent pane
   */
  private final AddDSPane myParent;

  /**
   * The constructor
   *
   * @param ds
   *          - dataset to be processed.
   * @param mp
   *          - JTextPane on which messages will be displayed.
   */
  public ValidityWorker(DataSet ds, AddDSPane aParent) {
    dataset = ds;
    myParent = aParent;
  }

  /**
   * Checks for correctess of dataset
   *
   * @return
   *
   * @throws Exception
   */
  @Override
  protected Integer doInBackground() throws Exception {
    // The number of instances the word is found
    publish(formatMessage("Checking for duplicate datasets"));
    String message = myParent.isIncluded(dataset);
    if (message != null) {
      publish(formatError(message));
      return 1;
    }
    setProgress(1);
    publish(formatOK("No duplicates found."));
    File[] filesInInputDir = dataset.getInputPath().listFiles();
    setProgress(2);
    if (isCancelled()) { return 1; }
    if (dataset.getType() == DataSet.Type.GPS) {
      publish(formatMessage("Checking subfolders of input directory."));
      if (filesInInputDir == null) {
        publish(formatError("Input directory \"" + dataset.getInputPath().toString()
            + "\"does not exists"));
        return 1;
      }
      for (File aFile : filesInInputDir) {
        String dirName = aFile.getName();
        if (!aFile.isDirectory()) {
          publish(formatError("Wrong format of input directory: it contains not a directory file: \""
              + dirName + "\""));
          return 1;
        }
        if (!dirName.matches("\\d{3}")) {
          publish(formatError("Wrong format of input directory: subfolder \"" + dirName
              + "\" does not consist of three digits"));
          return 1;
        }
        if (isCancelled()) { return 1; }
      }
      SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd");
      df.setLenient(false);
      setProgress(15);

      final int estimate=filesInInputDir.length;
      int done=0;

      for (File aFile : filesInInputDir) {
        File[] files = new File(aFile + File.separator + "Trajectory").listFiles();
        for (File aDayFile : files) {
          if (!aDayFile.isFile()) {
            publish(formatError("Wrong format of input directory: "
                + " There are folders in the subfolder of the user " + aFile.getName() + "."));
            return 1;
          }
          String firstEight = aDayFile.getName().substring(0, 8);
          try {
            if (!Character.isDigit(firstEight.charAt(7))) { throw new ParseException(null, 0); }
            df.parse(firstEight);
          }
          catch (ParseException ex) {
            publish(formatError("Wrong format of input directory: " + "The user " + aFile.getName()
                + " has a filename whose first eight characters are not a valid date."));
            return 1;
          }
          if (isCancelled()) { return 1; }
        }
        done++;
        setProgress((int)(2+18*done/estimate));
      }
      setProgress(20);
      publish(formatOK("Only correct subfolders found."));
    }
    else {// dataSet.getType==DataSet.Type.SP
      publish(formatMessage("Checking files in the input directory."));
      for (File aFile : filesInInputDir) {
        if (!aFile.isFile()) {
          publish(formatError("Wrong format of input directory: file \"" + aFile.toString()
              + "\" is not a regular file"));
          return 1;
        }
        if (isCancelled()) { return 1; }
      }
      setProgress(10);
      for (File aFile : filesInInputDir) {
        String fileName = aFile.getName().substring(0, 3);
        if (!fileName.matches("\\d{3}")) {
          publish(formatError("Wrong format of input directory: file \"" + aFile.getName()
              + "\" does not start with three digits"));
          return 1;
        }
        if (isCancelled()) { return 1; }
      }
      setProgress(20);
      publish(formatOK("Only correct files found."));
    }
    StringBuilder sb = new StringBuilder();
    double j = 1.0;
    UserStatistics stat;
    if (dataset.getType() == DataSet.Type.GPS) {
      publish(formatMessage("Reading GPS dataset and generating statistics."));
      for (File aUser : filesInInputDir) {
        stat = new UserStatistics();
        stat.setUserID(aUser.getName());
        String[] files = new File(aUser + "/Trajectory").list();
        TreeMap<String, LinkedList<String>> map = new TreeMap<String, LinkedList<String>>();
        for (int i = 0; i < files.length; i++) {
          String date = files[i].substring(0, 8);
          if (map.containsKey(date))
            map.get(date).add(files[i]);
          else {
            LinkedList<String> filelist = new LinkedList<String>();
            filelist.add(files[i]);
            map.put(date, filelist);
          }
          if (isCancelled()) { return 1; }
        }
        stat.setNumberOfDays(map.size());
        try {
          for (Map.Entry<String, LinkedList<String>> aDay : map.entrySet()) {
            int numberPointsInTheDay = 0;
            for (String aFile : aDay.getValue()) {
              BufferedReader br = new BufferedReader(new FileReader(aUser + "/Trajectory/" + aFile));
              for (int k = 0; k < 6; k++)
                br.readLine();
              // String aLine;
              while (br.readLine() != null) {
                numberPointsInTheDay++;
              }
              br.close();
              if (isCancelled()) { return 1; }
            }
            stat.setNumberOfPoints(stat.getNumberOfPoints() + numberPointsInTheDay);
            if (stat.getMaxNumberOfPointsInADay() < numberPointsInTheDay) {
              stat.setMaxNumberOfPointsInADay(numberPointsInTheDay);
            }
            if (stat.getMinNumberOfPointsInADay() > numberPointsInTheDay) {
              stat.setMinNumberOfPointsInADay(numberPointsInTheDay);
            }
          }
        }
        catch (Exception e) {
          publish(formatError(e.getMessage()));
          return 1;
        }
        sb.append(stat.toString());
        setProgress((int) ((j / filesInInputDir.length) * 80 + 20));
        j++;
      }
    }
    else {
      publish(formatMessage("Reading SP dataset and generating statistics."));
      for (File aUser : filesInInputDir) {
        stat = new UserStatistics();
        stat.setUserID(aUser.getName().substring(0, 3));
        int numOfDays = 0;
        int numOfPoints = 0;
        int maxNumOfPointsInADay = 0;
        int minNumOfPointsInADay = Integer.MAX_VALUE;
        try {
          BufferedReader bf = new BufferedReader(new FileReader(aUser));
          String aLine;
          while ((aLine = bf.readLine()) != null) {
            numOfDays++;
            String[] fields = aLine.split(" ");
            int numberOfPointsInADay = fields.length / 3 - 1;
            numOfPoints += numberOfPointsInADay;
            if (maxNumOfPointsInADay < numberOfPointsInADay) {
              maxNumOfPointsInADay = numberOfPointsInADay;
            }
            if (minNumOfPointsInADay > numberOfPointsInADay) {
              minNumOfPointsInADay = numberOfPointsInADay;
            }
          }
          bf.close();
          if (isCancelled()) { return 1; }
        }
        catch (Exception e) {
          publish(formatError(e.getMessage()));
          return 1;
        }
        stat.setNumberOfDays(numOfDays);
        stat.setNumberOfPoints(numOfPoints);
        stat.setMaxNumberOfPointsInADay(maxNumOfPointsInADay);
        stat.setMinNumberOfPointsInADay(minNumOfPointsInADay);
        sb.append(stat.toString());
        setProgress((int) ((j / filesInInputDir.length) * 80 + 20));
        j++;
      }
    }
    File dir = new File(dataset.getOutputPath().toString() + "/Stats");
    if (!dir.exists()) {
      dir.mkdirs();
    }
    BufferedWriter bw ;
    if (dataset.getType() == DataSet.Type.GPS) {
      File dira = new File(dataset.getOutputPath().toString() + "/Stats/SourceDataAndStayPoints");
      if (!dira.exists()) {
        dira.mkdirs();
      }
      bw = new BufferedWriter(new FileWriter(dira + "/SourceData.txt"));
    }
    else{
      bw = new BufferedWriter(new FileWriter(dir + "/StayPoints.txt"));
    }
    bw.write(sb.toString());
    bw.close();
    publish(formatOK("Finished."));
    setProgress(100);
    return 0;
  }

  /**
   * @param chunks
   */
  @Override
  protected void process(final List<String> chunks) {
    // Updates the messages text area
    for (final String string : chunks) {
      addMessage(string);
    }
  }

}
