package lu.uni.minus.utils.roi;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lu.uni.minus.preferences.DataSet;
import lu.uni.minus.utils.TextPaneWorker;

import weka.core.Attribute;
import weka.core.DenseInstance;
import weka.core.Instance;
import weka.core.Instances;
import weka.filters.Filter;

/**
 *
 * @author Piotr Kordy
 */
public class SPClusteringWorker extends TextPaneWorker
{
  /** The dataset to be processed */
  private final DataSet dataset;
  private final List<String> users;
  private int done;
  private final int percentage;
  private final int lowerK;
  private final int upperK;
  private double minLatitude = Double.MAX_VALUE;
  private double minLongitude = Double.MAX_VALUE;
  private double maxLatitude = Double.MIN_VALUE;
  private double maxLongitude = Double.MIN_VALUE;
  private final String selectedPara;

  public SPClusteringWorker(final DataSet ds, List<String> aUsers, final int aPercentage,
      final int aLowerK, final int aUpperK, final String aSelectedPara) {
    dataset = ds;
    users = aUsers;
    percentage = aPercentage;
    lowerK = aLowerK;
    upperK = aUpperK;
    selectedPara = aSelectedPara;
  }

  @Override
  protected Integer doInBackground() throws Exception {
    List<DataPoint> dataPoints = loadData();
    if (dataPoints != null) {
      Instances wekaDataset = toWekaFormat(dataPoints);
      dataPoints = removeOutliersUsingLOF(wekaDataset);
      if (dataPoints != null) {
        publish(formatMessage("Clustering GPS points."));

        List<Cluster> clusters = cluster(150, dataPoints);
        publish(formatOK("The number of clusters is: " + clusters.size() + "\n"));
        clusters = multiPointClusters(clusters);
        if (clusters != null) {
          int noRoI = clusters.size();
          publish(formatOK("The number of clusters that contain more than one point each is:"
              + noRoI + "\n"));
          if (noRoI > 0) {
            StringBuilder sbStats = new StringBuilder();
            sbStats.append(minLatitude + "\n" + minLongitude + "\n" + maxLatitude + "\n"
                + maxLongitude + "\n" + clusters.size());
            outputRegionFromCluster(clusters, sbStats.toString());
            publish(formatOK("Done."));

          }
          else {
            publish(formatMessage("There are no valid RoI to output"));
          }
          setProgress(100);
          return new Integer(0);
        }
      }
    }
    return new Integer(-1);
  }

  private List<DataPoint> loadData() {
    done = 0;
    setProgress(done);
    List<DataPoint> result = new ArrayList<DataPoint>();
    publish(formatMessage("Reading source data..."));
    int i = 0;
    try {
      for (String user : users) {
        BufferedReader br = new BufferedReader(
            new FileReader(dataset.getSPFile(selectedPara, user)));
        String line;
        while ((line = br.readLine()) != null) {

          String[] fields = line.split(" ");
          for (int j = 0; j < Integer.parseInt(fields[1]); j++) {
            double lat = Double.parseDouble(fields[4 + 3 * j]);
            double lngt = Double.parseDouble(fields[5 + 3 * j]);
            result.add(new DataPoint(lat, lngt));

            if (lat < minLatitude) {
              minLatitude = lat;
            }
            if (lat > maxLatitude) {
              maxLatitude = lat;
            }
            if (lngt < minLongitude) {
              minLongitude = lngt;
            }
            if (lngt > maxLongitude) {
              maxLongitude = lngt;
            }
          }
        }
        br.close();
        if (isCancelled()) {
          publish(formatError("Cancelled"));
          return null;
        }
        i++;
        done = (int) (10 * i / users.size());
        setProgress((int) (done));
      }
      publish(formatOK("Finished reading data. Number of loaded GPS points: " + result.size()));
      done = 10;
      setProgress(done);
      return result;
    }
    catch (FileNotFoundException e) {
      publish(formatError(e.getMessage()));
    }
    catch (NumberFormatException e) {
      publish(formatError(e.getMessage()));
    }
    catch (IOException e) {
      publish(formatError(e.getMessage()));
    }
    return null;
  }

  /**
   * Converts array of data poinst into weka dataset of Instances
   *
   * @param dataPoints
   * @return
   */
  private Instances toWekaFormat(List<DataPoint> dataPoints) {
    ArrayList<Attribute> att = new ArrayList<Attribute>();
    att.add(new Attribute("Latitude"));
    att.add(new Attribute("Longitude"));
    Instances result = new Instances("generatedByMinUS", att, dataPoints.size());
    for (DataPoint point : dataPoints) {
      double[] values = new double[2];
      values[0] = point.getLatitude();
      values[1] = point.getLongitude();
      Instance inst = new DenseInstance(1, values);
      inst.setDataset(result);
      result.add(inst);
    }
    done++;
    setProgress(done);
    return result;
  }

  private List<DataPoint> removeOutliersUsingLOF(Instances wekaDataset) {
    publish(formatMessage("Removing outliers."));
    final ArrayList<DataPoint> result = new ArrayList<DataPoint>();
    LOF lofFilter = new LOF(this);
    // Values set according to the paper Enhancing data analysis with noise
    // removal
    lofFilter.setMinPointsLowerBound(new Integer(lowerK).toString());
    lofFilter.setMinPointsUpperBound(new Integer(upperK).toString());
    try {
      lofFilter.setInputFormat(wekaDataset);
      // Executing the filter on the data.
      Instances resultInstances = null;
      resultInstances = Filter.useFilter(wekaDataset, lofFilter);
      for (weka.core.Instance instance : resultInstances) {
        result.add(new DataPoint(instance.value(0), instance.value(1), instance.value(2)));
      }
      // Sorted in descending LOF order.
      Collections.sort(result, new Comparator<DataPoint>()
      {
        public int compare(DataPoint o1, DataPoint o2) {
          return Double.compare(o1.LOF, o2.LOF);
        }
      });
      int deleteIndex = Math.round((result.size() * (100 - percentage) * 0.01f));

      while (deleteIndex < result.size() && result.get(deleteIndex).LOF < 1) {
        deleteIndex++;
      }
      if (deleteIndex < result.size()) {
        result.subList(deleteIndex, result.size()).clear();
      }
      publish(formatOK("Removed outliers. Remaining GPS points:" + result.size()));
      done = 30;
      setProgress(done);

      return result;
    }
    catch (Exception e) {
      publish(formatError(e.getMessage()));
      return null;
    }
  }

  private List<Cluster> cluster(int minStopDist, List<DataPoint> dataPoints) {

    List<Cluster> clusters = new ArrayList<Cluster>();
    // init each datapoint as own cluster
    int clusterIDCountr = 0;
    for (DataPoint dataPoint : dataPoints) {
      HashSet<DataPoint> clusterPoints = new HashSet<DataPoint>();
      clusterPoints.add(dataPoint);
      clusters.add(new Cluster(clusterIDCountr++, dataPoint));
    }
    double doneIn = Double.MAX_VALUE;
    int estimate = 0;
    while (true) {
      // find the pair of clusters with min distance to each other
      double minDistance = Double.MAX_VALUE;
      Cluster clusterWithMinDistanceSource = null;
      Cluster clusterWithMinDistanceTarget = null;
      double currentDistance = 0d;
      for (Cluster sourceCluster : clusters) {
        for (Cluster targetCluster : clusters) {
          if (sourceCluster == targetCluster) {
            continue;
          }
          currentDistance = euclidianDistance(sourceCluster.getCentroid(),
              targetCluster.getCentroid());
          if (currentDistance <= minDistance) {
            minDistance = currentDistance;
            clusterWithMinDistanceSource = sourceCluster;
            clusterWithMinDistanceTarget = targetCluster;
          }
        }// end of while
        if (isCancelled()) {
          publish(formatError("Cancelled"));
          return null;
        }
      }// end of for
      // ************************* NEW stop condition *************************
      // if (minDistance >= minimalStopDistance) {
      double mindistanceInMeters = greatCircleDistance(clusterWithMinDistanceSource.getCentroid(),
          clusterWithMinDistanceTarget.getCentroid());
      if (estimate==0){
        estimate=(int)(minStopDist-mindistanceInMeters);
      }
      doneIn = Math.min(doneIn, (minStopDist - mindistanceInMeters));
      done = (int) (31 + (1 - (doneIn / estimate)) * 68);
//       System.out.println("done "+done+ " minStop - mind"+(minStopDist-minDistance)+ "doneIn "+doneIn+ " estimate"+ estimate);
      setProgress(Math.max(0,Math.min(done, 100)));
      //
      if (mindistanceInMeters >= minStopDist) { return clusters; }

      // add target to source and remove target cluster from list
      if (clusterWithMinDistanceSource != null && clusterWithMinDistanceTarget != null) {
        clusterWithMinDistanceSource.addPoints(clusterWithMinDistanceTarget);
        clusters.remove(clusterWithMinDistanceTarget);
      }
    }// end of while

  }

  private static double euclidianDistance(DataPoint pt1, DataPoint pt2) {
    double diffLat = pt1.getLatitude() - pt2.getLatitude();
    double diffLong = pt1.getLongitude() - pt2.getLongitude();

    return Math.sqrt(Math.pow(diffLat, 2) + Math.pow(diffLong, 2));
  }

  private static double greatCircleDistance(DataPoint pt1, DataPoint pt2) {

    double a_x_point = pt1.getLatitude();
    double a_y_point = pt1.getLongitude();
    double b_x_point = pt2.getLatitude();
    double b_y_point = pt2.getLongitude();
    Double R = new Double(6371);
    Double dlat = (b_x_point - a_x_point) * Math.PI / 180;
    Double dlon = (b_y_point - a_y_point) * Math.PI / 180;
    Double aDouble = Math.sin(dlat / 2) * Math.sin(dlat / 2) + Math.cos(a_x_point * Math.PI / 180)
        * Math.cos(b_x_point * Math.PI / 180) * Math.sin(dlon / 2) * Math.sin(dlon / 2);
    Double cDouble = 2 * Math.atan2(Math.sqrt(aDouble), Math.sqrt(1 - aDouble));
    // double d = Math.round((R * cDouble) * (double) 1000); // FIXME: DONE set
    // value from 10000 to 1000
    double d = (R * cDouble) * (double) 1000; // FIXME: DONE set value from
                                              // 10000 to 1000
    return d;

  }

  private void outputRegionFromCluster(List<Cluster> clusters, String stats) {
    publish(formatMessage("Outputing RoIs...\n"));
    StringBuilder sb = new StringBuilder();
    Map<Integer, DataPoint> minLatLongs = new HashMap<Integer, DataPoint>();
    Map<Integer, DataPoint> maxLatLongs = new HashMap<Integer, DataPoint>();
    Map<Integer, Integer> numberOfPointsInCluster = new HashMap<Integer, Integer>();

    Set<Integer> clusterIDs = new HashSet<Integer>();

    for (Cluster cluster : clusters) {
      clusterIDs.add(cluster.id);
      // store min lat long for each cluster
      minLatLongs.put(cluster.id, cluster.getMinPoint());
      // store max lat long for each cluster
      maxLatLongs.put(cluster.id, cluster.getMaxPoint());

      numberOfPointsInCluster.put(cluster.id, cluster.clusterPoints.size());
    }

    StringBuilder sb2 = new StringBuilder();
    for (int i = 0; i < users.size(); i++) {
      if (i == users.size() - 1) {
        sb2.append(users.get(i));
      }
      else {
        sb2.append(users.get(i) + "_");
      }
    }
    sb2.append("-" + percentage + "_" + lowerK + "_" + upperK + ".txt");
    String fileName = sb2.toString();

    BufferedWriter bw = null;
    File outputFolder = dataset.createRoIDir(selectedPara);
    try {
      bw = new BufferedWriter(new FileWriter(new File(outputFolder + "/" + fileName)));
      int clusterRanking = 0;
      for (int clusterID : clusterIDs) {
        sb.append(clusterRanking++);
        sb.append(" ");
        double minLatitudeThisCluster = minLatLongs.get(clusterID).getLatitude();
        /*
         * if(minLatitudeThisCluster < minLatitude) { minLatitude =
         * minLatitudeThisCluster; }
         */
        sb.append(minLatitudeThisCluster);
        sb.append(" ");
        double minLongitudeThisCluster = minLatLongs.get(clusterID).getLongitude();
        /*
         * if(minLongitudeThisCluster < minLongitude) { minLongitude =
         * minLongitudeThisCluster; }
         */
        sb.append(minLongitudeThisCluster);
        sb.append(" ");
        double maxLatitudeThisCluster = maxLatLongs.get(clusterID).getLatitude();
        /*
         * if(maxLatitudeThisCluster > maxLatitude) { maxLatitude =
         * maxLatitudeThisCluster; }
         */
        sb.append(maxLatitudeThisCluster);
        sb.append(" ");
        double maxLongitudeThisCluster = maxLatLongs.get(clusterID).getLongitude();
        /*
         * if(maxLongitudeThisCluster > maxLongitude) { maxLongitude =
         * maxLongitudeThisCluster; }
         */
        sb.append(maxLongitudeThisCluster);
        sb.append("\n");
        /*
         * sb.append(" "); sb.append(numberOfPointsInCluster.get(clusterID));
         * sb.append(" "); sb.append(regionDiagonal);
         */
      }
      bw.write(sb.toString().trim());
      bw.close();
      publish(formatOK("Finished outputing RoIs...\n"));
    }
    catch (IOException e1) {
      publish(formatError(e1.getMessage()));
    }

    /*
     * StringBuilder sb3 = new StringBuilder(); sb3.append(minLatitude + "\n" +
     * minLongitude + "\n" + maxLatitude + "\n" + maxLongitude + "\n" +
     * clusters.size());
     */
    publish(formatMessage("Outputing stats of the RoIs..."));
    File statOutputFolder = dataset.createStatRoIDir(selectedPara);
    try {
      bw = new BufferedWriter(new FileWriter(statOutputFolder + "/" + fileName));
      bw.write(stats);
      bw.close();
      publish(formatOK("Finished outputing stats of the RoIs...\n"));
    }
    catch (IOException e1) {
      publish(formatError(e1.getMessage()));
    }
  }

  public List<Cluster> multiPointClusters(List<Cluster> clusters) {
    /** we have as many centroids as clusters and each centroid had 2 attributes */
    List<Cluster> multiPointClusters = new ArrayList<Cluster>();
    for (Cluster cluster : clusters) {
      if (cluster.clusterPoints.size() > 1) {
        multiPointClusters.add(cluster);
      }
    }
    return multiPointClusters;
  }

  /**
   * @param chunks
   */
  @Override
  protected void process(final List<String> chunks) {
    // Updates the messages text area
    for (final String string : chunks) {
      addMessage(string);
    }
  }

}
