package lu.uni.minus.utils.sp;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeMap;

import lu.uni.minus.preferences.DataSet;
import lu.uni.minus.utils.TextPaneWorker;
import lu.uni.minus.utils.UserStatistics;

/**
 *
 * @author Piotr Kordy
 */
public class SPDetectWorker extends TextPaneWorker
{
  /** The dataset to be processed */
  private final DataSet dataset;
  private final ArrayList<String> users;
  private int estimate;
  private final String distance;
  private final double distanceDouble;
  private final String time;
  private final double timeDouble;
  private final String merge;
  private final double mergeDouble;
  private int done;

  public SPDetectWorker(DataSet ds, ArrayList<String> aUsers, String distance, String time,
      String merge) {
    dataset = ds;
    users = aUsers;
    this.distance = distance;
    distanceDouble = Double.parseDouble(distance);
    this.time = time;
    timeDouble = Double.parseDouble(time);
    this.merge = merge;
    mergeDouble = Double.parseDouble(merge);
  }

  @Override
  protected Integer doInBackground() throws Exception {
    done = 0;
    estimate = estimateWork();
    setProgress(0);
    doAllUsers();
    setProgress(100);
    return new Integer(0);
  }

  /**
   * Estimate number of files needed to be processed
   *
   * @return number of files
   */
  private int estimateWork() {
    publish(formatMessage("Estimating needed time."));
    int result = 0;
    for (String user : users) {
      result = result + dataset.getUserInDir(user).list().length;
    }
    return result;
  }

  private void doAllUsers() {
    StringBuilder sb = new StringBuilder();
    for (String user : users) {
      UserStatistics stat = doUser(user);
      if (stat == null) {
        publish(formatError("Cancelled."));
        return;
      }
      sb.append(stat.toString());
    }
    try {
      File statOutDir = dataset.createOutputSPDir();
      BufferedWriter bw = new BufferedWriter(new FileWriter(statOutDir + File.separator
          + "StayPoints-" + distance + "_" + time + "_" + merge + ".txt", true));
      bw.write(sb.toString());
      bw.close();
      publish(formatOK("Done"));
    }
    catch (IOException e) {
      publish(formatError("Problem writing stats:" + e.getMessage()));
    }
  }

  private UserStatistics doUser(String user) {
    // out/StayPoints/dist_time_merge/user.txt
    setProgress((int) (100 * done / estimate));
    UserStatistics result = new UserStatistics();
    result.setUserID(user);
    TreeMap<String, ArrayList<String>> fileMap = new TreeMap<String, ArrayList<String>>();
    publish(formatMessage("Processing files of the user: \"" + user + "\"\n"));
    for (String file : dataset.getUserInDir(user).list()) {
      String date = file.substring(0, 8);
      if (fileMap.containsKey(date))
        fileMap.get(date).add(file);
      else {
        ArrayList<String> fileList = new ArrayList<String>();
        fileList.add(file);
        fileMap.put(date, fileList);
      }
    }
    result.setNumberOfDays(fileMap.size());
    Set<String> days = fileMap.keySet();
    Iterator<String> itr = days.iterator();
    ArrayList<ArrayList<StayPoint>> allSP = new ArrayList<ArrayList<StayPoint>>();
    while (itr.hasNext()) {
      final String date = itr.next();
      final ArrayList<String> fileList = fileMap.get(date);
      ArrayList<StayPoint> dateSP = doDate(date, fileList, user, result);
      if (dateSP == null) { return null; }
      allSP.add(dateSP);
      done = done + fileList.size();
      setProgress((int) (100 * done / estimate));
    }
    StringBuilder sb = new StringBuilder();
    Iterator<ArrayList<StayPoint>> spIterator = allSP.iterator();
    for (String date : fileMap.keySet()) {
      sb.append(date + " ");
      ArrayList<StayPoint> spDay = spIterator.next();
      sb.append(spDay.size() + " " + getWeekBit(date) + " ");
      for (StayPoint staypoint : spDay) {
        sb.append(staypoint.getArrivalTime() + " " + staypoint.getLatitude() + " "
            + staypoint.getLongitude() + " ");
      }
      sb.append("\n");
    }
    File spOutDir = dataset.createSPDir(distance, time, merge);
    BufferedWriter bw;
    try {
      bw = new BufferedWriter(new FileWriter(spOutDir + File.separator + user + ".txt"));
      bw.write(sb.toString());
      bw.close();
    }
    catch (IOException e) {
      publish(formatError(e.getMessage()));
    }
    return result;
  }

  private ArrayList<StayPoint> doDate(final String date, final ArrayList<String> fileList,
      String user, UserStatistics stat) {
    final ArrayList<GPSPoint> gpsPoints = new ArrayList<GPSPoint>();

    for (String file : fileList) {
      BufferedReader br;
      try {
        br = new BufferedReader(new FileReader(new File(dataset.getUserInDir(user) + File.separator
            + file)));
        for (int k = 0; k < 6; k++) {// skipping initial information
          br.readLine();
        }
        String line;
        while ((line = br.readLine()) != null) {
          String[] fields = line.split(",");
          double lat = Double.parseDouble(fields[0]);
          double lon = Double.parseDouble(fields[1]);
          String[] time = fields[6].split(":");
          int abstime = Integer.parseInt(time[0]) * 3600 + Integer.parseInt(time[1]) * 60
              + Integer.parseInt(time[2]);
          gpsPoints.add(new GPSPoint(lat, lon, abstime, date));
        }
        br.close();
        if (isCancelled()) { return null; }
      }
      catch (FileNotFoundException e) {
        publish(formatError(e.getMessage()));
      }
      catch (IOException e) {
        publish(formatError(e.getMessage()));
      }
    }
    ArrayList<StayPoint> result = detectStayPoints(gpsPoints);

    stat.setNumberOfPoints(stat.getNumberOfPoints() + result.size());
    if (stat.getMaxNumberOfPointsInADay() < result.size()) {
      stat.setMaxNumberOfPointsInADay(result.size());
    }
    if (stat.getMinNumberOfPointsInADay() > result.size()) {
      stat.setMinNumberOfPointsInADay(result.size());
    }
    publish(formatOK("Calculated " + result.size() + " staypoints for the date \"" +pretty(date) + "\"."));
    return result;
  }

  /**
   * Main algorithm for stay point detection
   *
   * @param gpsPoints
   *          - sorted list of gps points for one day
   * @return list of stay points for one day
   */
  private ArrayList<StayPoint> detectStayPoints(ArrayList<GPSPoint> gpsPoints) {
    // assume gpsPoints is sorted by time
    // Collections.sort(gpsPoints, new Comparator<GPSPoint>()
    // {
    // public int compare(GPSPoint p1, GPSPoint p2) {
    // int a = p1.getTime();
    // int b = p2.getTime();
    // return a > b ? +1 : a < b ? -1 : 0;
    // }
    // });
    ArrayList<StayPoint> result = new ArrayList<StayPoint>(); // stay points
    if (gpsPoints.size() > 0) {
      int index = 0;
      GPSPoint lP = gpsPoints.get(index);
      int within = withinDistance(lP, index, gpsPoints);
      result.add(averageSP(index, within, gpsPoints));// adding first
      int lastAdded = within;
      while (index < gpsPoints.size()) {
        lP = gpsPoints.get(index);
        within = withinDistance(lP, index, gpsPoints);
        if (within > 0) {
          double timeInter = (double) (gpsPoints.get(index + within).getTime() - gpsPoints.get(
              index).getTime());
          if (timeInter > this.timeDouble) {
            result.add(averageSP(index, index + within, gpsPoints));
            lastAdded = index + within;
          }
        }
        index = index + within + 1;
      }
      if (lastAdded < (gpsPoints.size() - 1)) {// adding last
        index = gpsPoints.size() - 1;
        while (index > lastAdded
            && gpsPoints.get(index).distanceTo(gpsPoints.get(gpsPoints.size() - 1)) < distanceDouble) {
          index--;
        }
        result.add(averageSP(index, gpsPoints.size() - 1, gpsPoints));
      }
      mergeStayPoints(result);

    }
    return result;

  }

  /**
   * Create staypoint as an average of GPS points
   *
   * @param lower
   *          index of lower GPS point
   * @param upper
   *          index of upper GPS point
   * @param gpsPoints
   *          array of GPS points
   * @return average staypoint
   */
  private StayPoint averageSP(int lower, int upper, ArrayList<GPSPoint> gpsPoints) {
    StayPoint s = new StayPoint(gpsPoints.get(lower));
    double latsum = 0.0;
    double lngtsum = 0.0;
    int numofpoints = upper - lower + 1;
    for (int k = lower; k <= upper; k++) {
      GPSPoint cur = gpsPoints.get(k);
      latsum += cur.getLatitude();
      lngtsum += cur.getLongitude();
    }
    s.setLatitude(latsum / numofpoints);
    s.setLongitude(lngtsum / numofpoints);
    s.setLeavingTime(gpsPoints.get(upper).getTime());
    return s;
  }

  /**
   *
   * Check how many GPS poins are within distance
   *
   * @param p
   * @param index
   * @param gpsPoints
   * @return
   */
  private int withinDistance(GPSPoint p, int index, ArrayList<GPSPoint> gpsPoints) {
    int result = 0;
    while ((index + result + 1) < gpsPoints.size()
        && gpsPoints.get(index).distanceTo(gpsPoints.get(index + result + 1)) < distanceDouble) {
      result++;
    }
    return result;
  }

  private String getWeekBit(String date) {
    SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd");
    try {
      df.parse(date);
    }
    catch (ParseException e) {
      publish(formatError(e.getMessage()));
    }

    int dayOfWeek = df.getCalendar().get(Calendar.DAY_OF_WEEK);
    if (dayOfWeek < 6) {
      return "1";
    }
    else {
      return "0";
    }
  }

  /**
   * Possibly merge last and first staypoints
   *
   * @param staypoints
   */
  private void mergeStayPoints(ArrayList<StayPoint> staypoints) {
    // algorithm get zero stay point
    int last = staypoints.size() - 1;
    if (staypoints.size() == 2) {
      // merge two points
      if (staypoints.get(0).distanceTo(staypoints.get(last)) < mergeDouble) {
        StayPoint s = staypoints.get(0).merge(staypoints.get(last));
        staypoints.clear();
        staypoints.add(s);
      }
    } // algorithm get one stay point
    else if (staypoints.size() == 3) {
      if (staypoints.get(0).distanceTo(staypoints.get(1)) < mergeDouble) {
        // merge first two points
        StayPoint s = staypoints.get(0).merge(staypoints.get(1));
        staypoints.set(0, s);
        staypoints.remove(1);
        last = staypoints.size() - 1;
      }
      else if (staypoints.get(1).distanceTo(staypoints.get(last)) < mergeDouble) {
        // or merge last two points
        StayPoint s = staypoints.get(1).merge(staypoints.get(last));
        staypoints.set(last, s);
        staypoints.remove(1);
        last = staypoints.size() - 1;
      }

      // merge three points
      if (staypoints.size() == 2
          && staypoints.get(0).distanceTo(staypoints.get(last)) < mergeDouble) {
        StayPoint s = staypoints.get(0).merge(staypoints.get(last));
        staypoints.clear();
        staypoints.add(s);
      }
    } // algorithm get one than one stay point
    else if (staypoints.size() > 3) {
      if (staypoints.get(0).distanceTo(staypoints.get(1)) < mergeDouble) {
        // merge the first two stay points
        StayPoint s = staypoints.get(0).merge(staypoints.get(1));
        staypoints.set(0, s);
        staypoints.remove(1);
        last = staypoints.size() - 1;
      } // FIXME: DONE REMOVED else
      if (staypoints.get(last - 1).distanceTo(staypoints.get(last)) < mergeDouble) {
        // merge the last two stay points
        StayPoint s = staypoints.get(last - 1).merge(staypoints.get(last));
        staypoints.set(last - 1, s);
        staypoints.remove(last);
      }
    }
  }

  private String pretty(String date) {
    if (date.length() == 8) {
      return date.substring(0, 4) + "-" + date.substring(4, 6) + "-" + date.substring(6, 8);
    }
    else {
      return date;
    }
  }

  /**
   * @param chunks
   */
  @Override
  protected void process(final List<String> chunks) {
    // Updates the messages text area
    for (final String string : chunks) {
      addMessage(string);
    }
  }

}
