/*
 * ============================================================================
 *
 *       Filename:  carve_test.cc
 *
 *    Description:  Tests for Carve class
 *
 *        Version:  1.0
 *        Created:  10/09/2011 09:23:56 AM
 *       Revision:  none
 *       Compiler:  gcc
 *
 *         Author:  Piotr Kordy (),
 *        Company:
 *
 * ============================================================================
 */
#include <stdio.h>
#include "gtest/gtest.h"
#include "carve.h"

class CarveTest;
//wrapper class to give access to protected members
class UnitCarve : public Carve {
private:
public:
  friend class CarveTest;
  UnitCarve (Options* option):Carve(option){
  }
  UnitCarve ():Carve(){
  }
  void callOutDiss8(VectorBuff &result,const uintlong &noBits,
                      VectorBuff &mask, std::ostream &os,
                      const uintlong &pos, std::deque<uintlong> &lastPos,
                      uintlong lastVal){
    outDiss(result,(uint_8t) noBits,mask,os,(uint_8t)pos,lastPos,lastVal);
  }
  void callOutDiss16(VectorBuff &result,const uintlong &noBits,
                      VectorBuff &mask, std::ostream &os,
                      const uintlong &pos, std::deque<uintlong> &lastPos,
                      uintlong lastVal){
    outDiss(result,(uint_16t) noBits,mask,os,(uint_16t)pos,lastPos,lastVal);
  }
  void callOutDiss32(VectorBuff &result,const uintlong &noBits,
                      VectorBuff &mask, std::ostream &os,
                      const uintlong &pos, std::deque<uintlong> &lastPos,
                      uintlong lastVal){
    outDiss(result,(uint_32t) noBits,mask,os,(uint_32t)pos,lastPos,lastVal);
  }
  void callOutDiss64(VectorBuff &result,const uintlong noBits,
                      VectorBuff &mask, std::ostream &os,
                      const uintlong pos, std::deque<uintlong> &lastPos,
                      uintlong lastVal){
    outDiss(result,(uint_64t) noBits,mask,os,(uint_64t)pos,lastPos,lastVal);
  }
  void callInitLastDiffBuff(VectorBuff& buff, const uint_16t nBAddr){//size in bytes
    initLastDiffBuff(buff, nBAddr);
  }
  void callDissim8(VectorBuff lastDiffBuff, ArrayBuff &buff,
                       const uintlong noBits, VectorBuff result) {
    dissim(lastDiffBuff,buff,(uint_8t)noBits, result);
  }
  void callDissim16(VectorBuff lastDiffBuff, ArrayBuff &buff,
                       const uintlong noBits, VectorBuff result) {
    dissim(lastDiffBuff,buff,(uint_16t)noBits, result);
  }
  void callDissim32(VectorBuff lastDiffBuff, ArrayBuff &buff,
                       const uintlong noBits, VectorBuff result) {
    dissim(lastDiffBuff,buff,(uint_32t)noBits, result);
  }
  void callDissim64(VectorBuff lastDiffBuff, ArrayBuff &buff,
                       const uintlong noBits, VectorBuff result) {
    dissim(lastDiffBuff,buff,(uint_64t)noBits, result);
  }
  void callDissimL8(BaseList &zeroes, BaseList &ones,
                      const uintlong &noBits, VectorBuff result){
    dissimL(zeroes,ones,(uint_8t)noBits,result);
  }
  void callDissimL16(BaseList &zeroes, BaseList &ones,
                      const uintlong &noBits, VectorBuff result){
    dissimL(zeroes,ones,(uint_16t)noBits,result);
  }
  void callDissimL32(BaseList &zeroes, BaseList &ones,
                      const uintlong &noBits, VectorBuff result){
    dissimL(zeroes,ones,(uint_32t)noBits,result);
  }
  void callDissimL64(BaseList &zeroes, BaseList &ones,
                      const uintlong &noBits, VectorBuff result){
    dissimL(zeroes,ones,(uint_64t)noBits,result);
  }
  uintlong callFillDiffBuff(ArrayBuff &destBuff, VectorBuff &mask,
                        std::vector<uint_32t> &noAttrValues,
                        std::vector<uintlong*> &order){
     return fillDiffBuff(destBuff, mask, noAttrValues, order);
  }
  void callFillCommBuff(vtPath &fileNames, const uintlong &pos,
                  ArrayBuff commBuff){
    fillCommBuff(fileNames,pos,commBuff);
  }
  void callSetOrder(AttValuesMap &attValuesMap,
                const ArrayBuff &commBuff,
                std::vector<uint_32t> &noAttrValues,
                std::vector<uintlong*> &order){
    setOrder(attValuesMap,commBuff, noAttrValues, order);
  }
  bool callReadFile(char* buff,const uint_32t &size, std::string &fileName,
                const uintlong &pos=0){
    return readFile(buff,size,fileName,pos);
  }
  void callExclude(ArrayBuff &buff, std::vector<uintlong*> &order,
                   VectorBuff &result){
    exclude(buff,order,result);
  }
  void callCheckFiles(vtPath &fileNames){
    checkFiles(fileNames);
  }
  void callCommonalities(std::vector<uint_32t> &noAttrValues,
                            std::vector<uintlong*> &order,
                            ArrayBuff &buff,
                            VectorBuff &result) {
    commonalities(noAttrValues,order,buff,result);
  }
  void callFormatedOut(VectorBuff buff, std::ostream &out){
    formatedOut(buff,out);
  }
  uint_32t callCalculateBlockSize(uintlong size,uint_32t noItems,const uint_16t nBAddr){
    return calculateBlockSize(size, noItems,nBAddr);
  }
  uint_16t callCalculateNoBytesForAddr(uintlong size){
    return calculateNoBytesForAddr(size);
  }
  void callInverseBits(char* p, uintlong size){
    inverseBits(p, size);
  }
  void callReorderBytes(char* p, uintlong size, uintlong noBytes)
  {
    reorderBytes(p, size, noBytes);
  }

};

class CarveTest : public testing::Test {
protected:
  vtPath fileNames;
  virtual void SetUp() { //run at start of each test
    fileNames.clear();
  }
  virtual void TearDown() {// run at the end of each test
    vtPath::iterator it;
    for (it=fileNames.begin(); it < fileNames.end(); it++ ) {
      remove(it->c_str());
    }
  }
  void createFile(std::string &name,std::string &content){
    FILE * pFile;
    pFile = fopen (name.c_str(), "wb");
    fwrite (content.c_str() , 1 , content.size(), pFile );
    fclose(pFile);
  }
  bool areEqual(const char * s1,const char* s2, unsigned long size){
    for ( uint_32t i=0u;i<size;i++ ) {
      if (*(s1+i)!=*(s2+i)) {
        //std::cout<<i<<" s"<<(char)*(s1+i)<<" s"<<(char)*(s2+i)<<" end\n";
        return false;
      }
    }
    return true;
  }
};

void init(BaseList &list){
  for(NodeType* it = list.begin(); it != list.end(); it=it->next ) {
    it->lastValue=0;
  }
}

TEST_F(CarveTest,setOrder){
  Options opt;
  char buff[1*5];
  uintlong* pointer = (uintlong*)buff;
  UnitCarve carv(&opt);
  std::vector<uintlong*>order;
  std::vector<uint_32t>fileIndexes;
  std::vector<uint_32t>noAttrValues;
  fileIndexes.push_back(0);
  fileIndexes.push_back(1);
  fileIndexes.push_back(3);
  AttValuesMap map;
  std::string s("ud");
  map[s]=&fileIndexes;
  ArrayBuff commBuff = {(char*)pointer,1,5};
  carv.callSetOrder(map,commBuff,noAttrValues,order);
  EXPECT_EQ(order[0],pointer);
  EXPECT_EQ(order[1],pointer+1);
  EXPECT_EQ(order[2],pointer+3);
  EXPECT_EQ(noAttrValues[0],3u);
}

TEST_F(CarveTest,outDiss){
  Options opt;
  std::deque<uintlong> lastPos;
  std::ostringstream os;
  UnitCarve carv(&opt);//noBits=20  noItems=3 blSize=8
  uint_8t reserve[160+5];
  VectorBuff result = {(char*)reserve,20*8};
  VectorBuff mask= {(char*)reserve+160,5};

  *(((uint_64t*)reserve)+0)=1;
  *(((uint_64t*)reserve)+1)=2;
  *(((uint_64t*)reserve)+2)=4;
  *(((uint_64t*)reserve)+3)=3;
  *(((uint_64t*)reserve)+4)=2;
  *(((uint_64t*)reserve)+5)=2;
  *(((uint_64t*)reserve)+6)=1;
  *(((uint_64t*)reserve)+7)=1;
  *(((uint_64t*)reserve)+8)=3;
  *(((uint_64t*)reserve)+9)=2;
  *(((uint_64t*)reserve)+10)=4;
  *(((uint_64t*)reserve)+11)=3;
  *(((uint_64t*)reserve)+12)=3;
  *(((uint_64t*)reserve)+13)=2;
  *(((uint_64t*)reserve)+14)=1;
  *(((uint_64t*)reserve)+15)=2;
  *(((uint_64t*)reserve)+16)=1;
  *(((uint_64t*)reserve)+17)=2;
  *(((uint_64t*)reserve)+18)=1;
  *(((uint_64t*)reserve)+19)=1;
  std::string expected="(37-39)\n(33-35)\n(28-30)\n(21-28)\n(17-21)\n(13-16)\n\
(12-13)\n(8-12)\n(5-7)\n(4-5)\n";
  memcpy(mask.p,"\
\x0F\xCC\xCC\xAA\x55",mask.sizeX);
  os.str("");
  lastPos.clear();
  carv.callOutDiss64(result, 20ul, mask, os,0ul,lastPos,0ul);
  EXPECT_EQ(os.str(),expected);

  result.sizeX = 20*4;
  *(((uint_32t*)reserve)+0)=1;
  *(((uint_32t*)reserve)+1)=2;
  *(((uint_32t*)reserve)+2)=4;
  *(((uint_32t*)reserve)+3)=3;
  *(((uint_32t*)reserve)+4)=2;
  *(((uint_32t*)reserve)+5)=2;
  *(((uint_32t*)reserve)+6)=1;
  *(((uint_32t*)reserve)+7)=1;
  *(((uint_32t*)reserve)+8)=3;
  *(((uint_32t*)reserve)+9)=2;
  *(((uint_32t*)reserve)+10)=4;
  *(((uint_32t*)reserve)+11)=3;
  *(((uint_32t*)reserve)+12)=3;
  *(((uint_32t*)reserve)+13)=2;
  *(((uint_32t*)reserve)+14)=1;
  *(((uint_32t*)reserve)+15)=2;
  *(((uint_32t*)reserve)+16)=1;
  *(((uint_32t*)reserve)+17)=2;
  *(((uint_32t*)reserve)+18)=1;
  *(((uint_32t*)reserve)+19)=1;
  os.str("");
  lastPos.clear();
  carv.callOutDiss32(result, 20, mask, os,0,lastPos,0);
  EXPECT_EQ(os.str(),expected);

  result.sizeX = 20*2;
  *(((uint_16t*)reserve)+0)=1;
  *(((uint_16t*)reserve)+1)=2;
  *(((uint_16t*)reserve)+2)=4;
  *(((uint_16t*)reserve)+3)=3;
  *(((uint_16t*)reserve)+4)=2;
  *(((uint_16t*)reserve)+5)=2;
  *(((uint_16t*)reserve)+6)=1;
  *(((uint_16t*)reserve)+7)=1;
  *(((uint_16t*)reserve)+8)=3;
  *(((uint_16t*)reserve)+9)=2;
  *(((uint_16t*)reserve)+10)=4;
  *(((uint_16t*)reserve)+11)=3;
  *(((uint_16t*)reserve)+12)=3;
  *(((uint_16t*)reserve)+13)=2;
  *(((uint_16t*)reserve)+14)=1;
  *(((uint_16t*)reserve)+15)=2;
  *(((uint_16t*)reserve)+16)=1;
  *(((uint_16t*)reserve)+17)=2;
  *(((uint_16t*)reserve)+18)=1;
  *(((uint_16t*)reserve)+19)=1;
  os.str("");
  lastPos.clear();
  carv.callOutDiss16(result, 20, mask, os,0,lastPos,0);
  EXPECT_EQ(os.str(),expected);

  result.sizeX=20*1;
  *(((uint_8t*)reserve)+0)=1;
  *(((uint_8t*)reserve)+1)=2;
  *(((uint_8t*)reserve)+2)=4;
  *(((uint_8t*)reserve)+3)=3;
  *(((uint_8t*)reserve)+4)=2;
  *(((uint_8t*)reserve)+5)=2;
  *(((uint_8t*)reserve)+6)=1;
  *(((uint_8t*)reserve)+7)=1;
  *(((uint_8t*)reserve)+8)=3;
  *(((uint_8t*)reserve)+9)=2;
  *(((uint_8t*)reserve)+10)=4;
  *(((uint_8t*)reserve)+11)=3;
  *(((uint_8t*)reserve)+12)=3;
  *(((uint_8t*)reserve)+13)=2;
  *(((uint_8t*)reserve)+14)=1;
  *(((uint_8t*)reserve)+15)=2;
  *(((uint_8t*)reserve)+16)=1;
  *(((uint_8t*)reserve)+17)=2;
  *(((uint_8t*)reserve)+18)=1;
  *(((uint_8t*)reserve)+19)=1;
  os.str("");
  lastPos.clear();
  carv.callOutDiss8(result, 20, mask, os,0,lastPos,0);
  EXPECT_EQ(os.str(),expected);
}

TEST_F(CarveTest,dissim){
  Options opt;
  BaseList ones;
  BaseList zeroes;
  UnitCarve carv(&opt);//noBits=20  noItems=3 blSize=8
  char reserve[8*3+20*8+3*8+20*8];
  ArrayBuff buff={reserve,8,3};
  VectorBuff res={reserve+8*3,20*8};
  VectorBuff lastDA={res.p+20*8,3*(3-1)/2*8};
  char* expected=lastDA.p+3*8;
  NodeType nodeArr[3]={
  NodeType((uint_8t*)buff.p),
  NodeType((uint_8t*)buff.p+8),
  NodeType((uint_8t*)buff.p+16)};
  memcpy(buff.p,"\
\x0C\x0F\x0F\x00\xCC\x00\x00\xF0\
\x35\x33\x03\x00\xCC\x00\x00\xF0\
\x2A\xA2\x0A\x00\x0F\x00\x00\x00",buff.sizeX*buff.sizeY);
// 0x0C0F0F00CC0000F0
// 0x35330300CC0000F0
// 0x2AA20A000F000000
// 0000_1100 0000_1111 (0000)_1111 0000000011001100000000000000000011110000 0xbfffeb14
// 0011_0101 0011_0011 (0000)_0011 0000000011001100000000000000000011110000 0xbfffeb24
// 0010_1010 1010_0010 (0000)_1010 0000000000001111000000000000000000000000 0xbfffeb34
// 3212 1211 2211 3243  ----- 1243
// 7654 3210 FEDC BA98
  *(((uint_64t*)expected)+0)=1;
  *(((uint_64t*)expected)+1)=2;
  *(((uint_64t*)expected)+2)=4;
  *(((uint_64t*)expected)+3)=3;
  *(((uint_64t*)expected)+4)=2;
  *(((uint_64t*)expected)+5)=2;
  *(((uint_64t*)expected)+6)=1;
  *(((uint_64t*)expected)+7)=1;
  *(((uint_64t*)expected)+8)=3;
  *(((uint_64t*)expected)+9)=2;
  *(((uint_64t*)expected)+10)=4;
  *(((uint_64t*)expected)+11)=3;
  *(((uint_64t*)expected)+12)=3;
  *(((uint_64t*)expected)+13)=2;
  *(((uint_64t*)expected)+14)=1;
  *(((uint_64t*)expected)+15)=2;
  *(((uint_64t*)expected)+16)=1;
  *(((uint_64t*)expected)+17)=2;
  *(((uint_64t*)expected)+18)=1;
  *(((uint_64t*)expected)+19)=1;
  carv.callInitLastDiffBuff(lastDA,8);
  carv.callDissim64(lastDA,buff,20,res);
  EXPECT_TRUE(areEqual(res.p,expected,160));
  zeroes.push_front(&nodeArr[0]);
  zeroes.push_front(&nodeArr[1]);
  zeroes.push_front(&nodeArr[2]);
  carv.callDissimL64(zeroes,ones,20,res);
  EXPECT_TRUE(areEqual(res.p,expected,160));

  *(((uint_32t*)expected)+0)=1;
  *(((uint_32t*)expected)+1)=2;
  *(((uint_32t*)expected)+2)=4;
  *(((uint_32t*)expected)+3)=3;
  *(((uint_32t*)expected)+4)=2;
  *(((uint_32t*)expected)+5)=2;
  *(((uint_32t*)expected)+6)=1;
  *(((uint_32t*)expected)+7)=1;
  *(((uint_32t*)expected)+8)=3;
  *(((uint_32t*)expected)+9)=2;
  *(((uint_32t*)expected)+10)=4;
  *(((uint_32t*)expected)+11)=3;
  *(((uint_32t*)expected)+12)=3;
  *(((uint_32t*)expected)+13)=2;
  *(((uint_32t*)expected)+14)=1;
  *(((uint_32t*)expected)+15)=2;
  *(((uint_32t*)expected)+16)=1;
  *(((uint_32t*)expected)+17)=2;
  *(((uint_32t*)expected)+18)=1;
  *(((uint_32t*)expected)+19)=1;
  lastDA.sizeX=3*4;
  carv.callInitLastDiffBuff(lastDA,4);
  res.sizeX=20*4;
  carv.callDissim32(lastDA,buff,20,res);
  EXPECT_TRUE(areEqual(res.p,expected,80));
  init(ones);
  init(zeroes);
  carv.callDissimL32(zeroes,ones,20,res);
  EXPECT_TRUE(areEqual(res.p,expected,80));

  *(((uint_16t*)expected)+0)=1;
  *(((uint_16t*)expected)+1)=2;
  *(((uint_16t*)expected)+2)=4;
  *(((uint_16t*)expected)+3)=3;
  *(((uint_16t*)expected)+4)=2;
  *(((uint_16t*)expected)+5)=2;
  *(((uint_16t*)expected)+6)=1;
  *(((uint_16t*)expected)+7)=1;
  *(((uint_16t*)expected)+8)=3;
  *(((uint_16t*)expected)+9)=2;
  *(((uint_16t*)expected)+10)=4;
  *(((uint_16t*)expected)+11)=3;
  *(((uint_16t*)expected)+12)=3;
  *(((uint_16t*)expected)+13)=2;
  *(((uint_16t*)expected)+14)=1;
  *(((uint_16t*)expected)+15)=2;
  *(((uint_16t*)expected)+16)=1;
  *(((uint_16t*)expected)+17)=2;
  *(((uint_16t*)expected)+18)=1;
  *(((uint_16t*)expected)+19)=1;
  lastDA.sizeX=3*2;
  carv.callInitLastDiffBuff(lastDA,2);
  res.sizeX=20*2;
  carv.callDissim16(lastDA,buff,20,res);
  EXPECT_TRUE(areEqual(res.p,expected,40));
  init(ones);
  init(zeroes);
  carv.callDissimL16(zeroes,ones,20,res);
  EXPECT_TRUE(areEqual(res.p,expected,40));

  *(((uint_8t*)expected)+0)=1;
  *(((uint_8t*)expected)+1)=2;
  *(((uint_8t*)expected)+2)=4;
  *(((uint_8t*)expected)+3)=3;
  *(((uint_8t*)expected)+4)=2;
  *(((uint_8t*)expected)+5)=2;
  *(((uint_8t*)expected)+6)=1;
  *(((uint_8t*)expected)+7)=1;
  *(((uint_8t*)expected)+8)=3;
  *(((uint_8t*)expected)+9)=2;
  *(((uint_8t*)expected)+10)=4;
  *(((uint_8t*)expected)+11)=3;
  *(((uint_8t*)expected)+12)=3;
  *(((uint_8t*)expected)+13)=2;
  *(((uint_8t*)expected)+14)=1;
  *(((uint_8t*)expected)+15)=2;
  *(((uint_8t*)expected)+16)=1;
  *(((uint_8t*)expected)+17)=2;
  *(((uint_8t*)expected)+18)=1;
  *(((uint_8t*)expected)+19)=1;
  lastDA.sizeX=3;
  carv.callInitLastDiffBuff(lastDA,1);
  res.sizeX=20;
  carv.callDissim8(lastDA,buff,20,res);
  EXPECT_TRUE(areEqual(res.p,expected,20));
  init(ones);
  init(zeroes);
  carv.callDissimL8(zeroes,ones,20,res);
  EXPECT_TRUE(areEqual(res.p,expected,20));
}

TEST_F(CarveTest,commDissOut){
  Options opt;
  BaseList ones;
  BaseList zeroes;
  UnitCarve carv(&opt);//noBits=20  noItems=3 blSize=8
  const uint_8t noItems = 4;
  char reserve[noItems*8+8+noItems*8+7+1+7];
  ArrayBuff buff={reserve,1,noItems};
  memcpy(buff.p,"\
\xbf\x00\x00\x00\x00\x00\x00\x00\
\x00\x00\x00\x00\x00\x00\x00\x00\
\x00\x00\x00\x00\x00\x00\x00\x00\
\xbf\x00\x00\x00\x00\x00\x00\x00\
" ,noItems*8);
  VectorBuff commRes={reserve+8*noItems,1};
  ArrayBuff diffBuff={commRes.p+8,1,noItems};
  VectorBuff diffRes= {diffBuff.p+noItems,7};
  VectorBuff lastDiff= {diffRes.p+7,1};
  VectorBuff diffResL= {lastDiff.p+1,7};
  std::vector<uintlong*>order;
  std::vector<uint_32t>fileIndexes1;
  std::vector<uint_32t>fileIndexes2;
  std::vector<uint_32t>noAttrValues;
  fileIndexes1.push_back(0);
  fileIndexes1.push_back(3);
  fileIndexes2.push_back(1);
  fileIndexes2.push_back(2);
  AttValuesMap map;
  std::string s1("val1");
  map[s1]=&fileIndexes1;
  std::string s2("val2");
  map[s2]=&fileIndexes2;
  carv.callSetOrder(map,buff,noAttrValues,order);
  carv.callCommonalities(noAttrValues,order,buff,commRes);
  carv.callExclude(buff,order,commRes);
  uintlong noBits =carv.callFillDiffBuff(diffBuff,commRes,noAttrValues,order);
  EXPECT_EQ(noBits,7u);
  NodeType nodeArr[2]={
  NodeType((uint_8t*)diffBuff.p),
  NodeType((uint_8t*)diffBuff.p+1)};
  zeroes.push_front(&nodeArr[0]);
  zeroes.push_front(&nodeArr[1]);
  carv.callInitLastDiffBuff(lastDiff, 1);
  carv.callDissimL8(zeroes,ones,noBits,diffRes);
  carv.callDissim8(lastDiff,diffBuff,noBits,diffResL);
  EXPECT_TRUE(areEqual(diffResL.p,diffRes.p,7));

}

TEST_F(CarveTest,dissim2){
  Options opt;
  BaseList ones;
  BaseList zeroes;
  UnitCarve carv(&opt);
  const uint_8t noBits = 6;
  const uint_8t noItems = 2;
  char reserve[noItems+noBits+1+noBits];
  ArrayBuff buff={reserve,1,noItems};
  VectorBuff res={reserve+noItems,noBits};
  VectorBuff lastDA={res.p+noBits,1};
  char* expected=lastDA.p+1;
  NodeType nodeArr[2]={
  NodeType((uint_8t*)buff.p),
  NodeType((uint_8t*)buff.p+1)};
  memcpy(buff.p,"\xbf\x00",noItems);
  *(((uint_8t*)expected)+0)=0;
  *(((uint_8t*)expected)+1)=0;
  *(((uint_8t*)expected)+2)=0;
  *(((uint_8t*)expected)+3)=0;
  *(((uint_8t*)expected)+4)=0;
  *(((uint_8t*)expected)+5)=0;
  carv.callInitLastDiffBuff(lastDA,1);
  carv.callDissim8(lastDA,buff,noBits,res);
  EXPECT_TRUE(areEqual(res.p,expected,6));
  zeroes.push_front(&nodeArr[0]);
  zeroes.push_front(&nodeArr[1]);
  carv.callDissimL8(zeroes,ones,noBits,res);
  EXPECT_TRUE(areEqual(res.p,expected,6));

}

TEST_F(CarveTest,fillDiffBuff){
  Options opt;
  UnitCarve carv(&opt);
  char reserve[32+24+8+24];
  char* buff=reserve;
  char* expected=reserve+32;
  VectorBuff res={reserve+32+24,8};
  ArrayBuff diffBuff={reserve+32+24+8,8,3};
  memcpy(buff,"\
\xFF\x49\x13\x20\xCC\xB6\xF0\xF0\
\xFF\x94\x17\x30\xCC\x96\xF0\xF0\
\x0B\xF4\x51\x7A\xF0\xD6\x0F\x0F\
\xC0\x45\xF0\xA8\x0F\x7F\xC7\x00",32);
  memcpy(expected,"\
\xCC\xF9\x10\xFF\x00\x00\x00\x00\
\x90\x37\xF8\xC0\x00\x00\x00\x00\
\x33\xCD\x5A\x0B\x00\x00\x00\x00",24); //reversed bytes but not bits
  memcpy(res.p,"\
\xFF\x00\xF0\x0F\xCC\xCC\xAA\x55",res.sizeX);//0FF000FF
  memcpy(diffBuff.p,"\
\x00\x00\x00\x00\x00\x00\x00\x00\
\x00\x00\x00\x00\x00\x00\x00\x00\
\x00\x00\x00\x00\x00\x00\x00\x00",24);
  std::vector<uintlong*>order;
  order.push_back((uintlong*)buff);
  order.push_back((uintlong*)buff+1);
  order.push_back((uintlong*)buff+3);
  order.push_back((uintlong*)buff+2);
  std::vector<uint_32t> attrVal;
  attrVal.push_back(2);
  attrVal.push_back(1);
  attrVal.push_back(1);
  uintlong noBits;
  noBits= carv.callFillDiffBuff(diffBuff, res, attrVal,order);
  EXPECT_EQ(noBits,32u);
  EXPECT_TRUE(areEqual(diffBuff.p,expected,24));
}

TEST_F(CarveTest,commonalities){
  Options opt;
  UnitCarve carv(&opt);
  std::vector<uintlong*>order;
  char reserve[24+8+8];
  ArrayBuff buff={reserve,8,3};
  char* expected=reserve+24;
  VectorBuff res={reserve+24+8,8};
  memcpy(buff.p,"\
\xCB\x99\x11\x20\xCB\x99\x11\x20\
\xC9\x99\x45\x20\xC9\x99\x45\x20\
\xC8\x99\x7F\x20\xC8\x99\x7F\x20",24);
  memcpy(expected,"\
\xFC\xFF\x81\xFF\xFC\xFF\x81\xFF",8);
  std::vector<uint_32t> attrVal;
  attrVal.push_back(3);
  order.push_back((uintlong*)buff.p);
  order.push_back((uintlong*)buff.p+1);
  order.push_back((uintlong*)buff.p+2);
  carv.callCommonalities(attrVal,order,buff,res);
  EXPECT_TRUE(areEqual(res.p,expected,8));
  attrVal.push_back(1);
  attrVal[0]=2;
  memcpy(buff.p,"\
\xF0\xF9\x15\x20\xF0\xF9\x15\x20\
\x00\x08\x15\x20\x00\x08\x15\x20\
\xF8\x99\x7F\x20\xF8\x99\x7F\x20",24);
  memcpy(expected,"\
\x0F\x0e\xFF\xFF\x0F\x0e\xFF\xFF",8);
  carv.callCommonalities(attrVal,order,buff,res);
  EXPECT_TRUE(areEqual(res.p,expected,8u));
  attrVal.push_back(1u);
  attrVal[0]=1u;
  memcpy(expected,"\
\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF",8);
  carv.callCommonalities(attrVal,order,buff,res);
  EXPECT_TRUE(areEqual(res.p,expected,8u));
}

TEST_F(CarveTest,exclude){
  Options opt;
  UnitCarve carv(&opt);
  char reserve[48+16+16];
  ArrayBuff buff={reserve,16,3};
  char* expected=reserve+48;
  VectorBuff res={reserve+48+16,16};
  memcpy(buff.p,"\
\xFF\xFF\x0F\x20\xFF\xFF\x0F\x20\xFF\xFF\x0F\x20\xFF\xFF\x0F\x20\
\x00\xFF\xF0\x67\x00\xFF\xF0\x67\x00\xFF\xF0\x67\x00\xFF\xF0\x67\
\xC8\xFF\x7F\x20\xC8\xFF\x7F\x20\xC8\xFF\x7F\x20\xC8\xFF\x7F\x20",48);
  memcpy(expected,"\
\xFF\x00\xFF\x00\xFF\x00\xFF\x00\xFF\x00\xFF\x00\xFF\x00\xFF\x00",16);
  memcpy(res.p,"\
\xFF\xFF\xFF\x00\xFF\xFF\xFF\x00\xFF\xFF\xFF\x00\xFF\xFF\xFF\x00",
         16);
  std::vector<uintlong*>order;
  order.push_back((uintlong*)buff.p);
  order.push_back((uintlong*)buff.p+1);
  order.push_back((uintlong*)buff.p+2);
  carv.callExclude(buff,order,res);
  EXPECT_TRUE(areEqual(res.p,expected,16));
}

TEST_F(CarveTest,calculateBlockSize){
  Options opt;
  opt.useLinear=false;
  UnitCarve carve(&opt);
  opt.memLimit=1000;
  EXPECT_EQ(carve.callCalculateBlockSize(10,10,0),16u);
  EXPECT_EQ(carve.callCalculateBlockSize(1,10,0),8u);
  EXPECT_EQ(carve.callCalculateBlockSize(10000,9,0),104u);
  opt.dissimilFile="te";
  EXPECT_EQ(carve.callCalculateBlockSize(10000,9,4),16u);
}

TEST_F(CarveTest,checkFiles){
  Options opt;
  UnitCarve carve(&opt);
  ASSERT_DEATH(carve.callCheckFiles(fileNames),
               "Project file must .*");
  fileNames.push_back("__tmp__file1");
  remove("__tmp__file1");
  ASSERT_DEATH(carve.callCheckFiles(fileNames),"Project file must .*");
  fileNames.push_back("__tmp__file2");
  std::string s1,s2;
  s1="__tmp__file1";
  s2="aabb";
  createFile(s1,s2);
  remove("__tmp__file2");
  ASSERT_DEATH(carve.callCheckFiles(fileNames),"File \"__tmp__file2\" specified.*");
  s1="__tmp__file2";
  s2="aaeaeuaeb";
  createFile(s1,s2);
  ASSERT_DEATH(carve.callCheckFiles(fileNames),"Not all files specified.*");
}

TEST_F(CarveTest,readFile){
  Options opt;
  char buff[9];
  UnitCarve carv(&opt);
  std::string s1("__tmp__file");
  std::string s2("123456789");
  createFile(s1,s2);
  carv.callReadFile(buff,9,s1);
  EXPECT_EQ(s2.compare(0,9,buff,9),0);
  remove(s1.c_str());
}

TEST_F(CarveTest,file_size){
  Options opt;
  UnitCarve carv(&opt);
  std::string s1("__tmp__file");
  std::string s2("123456789");
  createFile(s1,s2);
  EXPECT_EQ(boost::filesystem3::file_size("__tmp__file"),li_32(09));
  remove(s1.c_str());
}

TEST_F(CarveTest,reorderBytes){
  Options opt;
  UnitCarve carv(&opt);
  char buff[17]= "abcdefgh12345678";
  char expected[17]= "badcfehg21436587";
  const char* orig= "abcdefgh12345678";
  carv.callReorderBytes(buff,16,2);
  EXPECT_TRUE(areEqual(buff,expected,17));
  strncpy(buff,"abcdefgh12345678",17);
  strncpy(expected, "dcbahgfe43218765",17);
  carv.callReorderBytes(buff,16,4);
  EXPECT_TRUE(areEqual(buff,expected,17));
  strncpy(buff,"abcdefgh12345678",17);
  strncpy(expected,"hgfedcba87654321",17);
  carv.callReorderBytes(buff,16,8);
  EXPECT_TRUE(areEqual(buff,expected,17));
  strncpy(buff,"abcdefgh12345678",17);
  carv.callReorderBytes(buff,16,8);
  carv.callReorderBytes(buff,16,8);
  EXPECT_TRUE(areEqual(buff,orig,17));
  carv.callReorderBytes(buff,16,4);
  carv.callReorderBytes(buff,16,4);
  EXPECT_TRUE(areEqual(buff,orig,17));
  carv.callReorderBytes(buff,16,2);
  carv.callReorderBytes(buff,16,2);
  EXPECT_TRUE(areEqual(buff,orig,17));
}

TEST_F(CarveTest,inverseBits){
  Options opt;
  UnitCarve carv(&opt);
  unsigned char buff[9]= {
   0b00001111,0b10100101,0b10101010,0b10100000,
   0b11101111,0b11101011,0b00001110,0b10010110,
   0b00111100
  };
  unsigned char expected[9]= {
   0b11110000,0b10100101,0b01010101,0b00000101,
   0b11110111,0b11010111,0b01110000,0b01101001,
   0b00111100
  };
  carv.callInverseBits((char*)buff,8);
  EXPECT_TRUE(areEqual((char*)buff,(char*)expected,9));
  unsigned char buff2[9]= {
   0b10000000,0b01000000,0b00100000,0b00010000,
   0b00001000,0b00000100,0b00000010,0b00000001,
   0b00111100
  };
  unsigned char expected2[9]= {
   0b00000001,0b00000010,0b00000100,0b00001000,
   0b00010000,0b00100000,0b01000000,0b10000000,
   0b00111100
  };
  carv.callInverseBits((char*)buff2,7);
  EXPECT_TRUE(areEqual((char*)buff2,(char*)expected2,9));
}

TEST_F(CarveTest,formatedOut){
  Options opt;
  std::ostringstream out;
  UnitCarve carv(&opt);
  char buff[5];
  VectorBuff pBuff={buff,5};
  memcpy(buff,"\x00\xFF\x01\x05\x00",5);
  std::cout<<buff;
  opt.outputFormat=HEX_OUTPUT;
  opt.separateBytes = true;
  carv.callFormatedOut(pBuff,out);
  memcpy(buff,"\x00\xFF\x01\x05\x00",5);
  EXPECT_EQ(out.str().compare("00 FF 01 05 00"),0);
  out.str("");
  opt.separateBytes = false;
  carv.callFormatedOut(pBuff,out);
  memcpy(buff,"\x00\xFF\x01\x05\x00",5);
  EXPECT_EQ(out.str().compare("00FF010500"),0);
  out.str("");
  opt.outputFormat=OCT_OUTPUT;
  opt.separateBytes = true;
  carv.callFormatedOut(pBuff,out);
  memcpy(buff,"\x00\xFF\x01\x05\x00",5);
  EXPECT_EQ(out.str().compare("000 377 001 005 000"),0);
  out.str("");
  opt.separateBytes = false;
  carv.callFormatedOut(pBuff,out);
  memcpy(buff,"\x00\xFF\x01\x05\x00",5);
  EXPECT_EQ(out.str().compare("000377001005000"),0);
  out.str("");
  opt.outputFormat=DEC_OUTPUT;
  opt.separateBytes = true;
  carv.callFormatedOut(pBuff,out);
  memcpy(buff,"\x00\xFF\x01\x05\x00",5);
  EXPECT_EQ(out.str().compare("000 255 001 005 000"),0);
  out.str("");
  opt.separateBytes = false;
  carv.callFormatedOut(pBuff,out);
  memcpy(buff,"\x00\xFF\x01\x05\x00",5);
  EXPECT_EQ(out.str().compare("000255001005000"),0);
  out.str("");
  memcpy(buff,"\x80\x40\x10\x08\x02",5);
  opt.outputFormat=BIN_OUTPUT;
  opt.separateBytes = true;
  carv.callFormatedOut(pBuff,out);
  memcpy(buff,"\x80\x40\x10\x08\x02",5);
  EXPECT_EQ(out.str().compare("10000000 01000000 00010000 00001000 00000010"),0);
  out.str("");
  opt.separateBytes = false;
  carv.callFormatedOut(pBuff,out);
  EXPECT_EQ(out.str().compare("1000000001000000000100000000100000000010"),0);
}
