//==============================================================================
//	
//	This file is originally taken from PRISM. We take and modify it in accordance
//  with our needs.
//	
//==============================================================================
#include <stdio.h>
#include <stdlib.h>
#include <util.h>
#include <cudd.h>
#include "dd_basics.h"

//------------------------------------------------------------------------------

DdNode *DD_Create(DdManager *ddman)
{
	DdNode *res;

	res = Cudd_addConst(ddman, 0);
	Cudd_Ref(res);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_Constant(DdManager *ddman, double value)
{
	DdNode *res;

	res = Cudd_addConst(ddman, value);
	Cudd_Ref(res);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_PlusInfinity(DdManager *ddman)
{
	DdNode *res;

	res = Cudd_ReadPlusInfinity(ddman);
	Cudd_Ref(res);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_MinusInfinity(DdManager *ddman)
{
	DdNode *res;

	res = Cudd_ReadMinusInfinity(ddman);
	Cudd_Ref(res);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_Var(DdManager *ddman, int i)
{
	DdNode *res;

	res = Cudd_addIthVar(ddman, i);
	Cudd_Ref(res);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_Not(DdManager *ddman, DdNode *dd)
{
	DdNode *res;

	res = Cudd_addCmpl(ddman, dd);
	Cudd_Ref(res);
	Cudd_RecursiveDeref(ddman, dd);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_Or(DdManager *ddman, DdNode *dd1, DdNode *dd2)
{
	DdNode *res;

	res = Cudd_addApply(ddman, Cudd_addOr, dd1, dd2);
	Cudd_Ref(res);
	Cudd_RecursiveDeref(ddman, dd1);
	Cudd_RecursiveDeref(ddman, dd2);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_And(DdManager *ddman, DdNode *dd1, DdNode *dd2)
{
	return DD_Not(ddman, DD_Or(ddman, DD_Not(ddman, dd1), DD_Not(ddman, dd2)));
}

//------------------------------------------------------------------------------

DdNode *DD_Xor(DdManager *ddman, DdNode *dd1, DdNode *dd2)
{
	DdNode *res;

	res = Cudd_addApply(ddman, Cudd_addXor, dd1, dd2);
	Cudd_Ref(res);
	Cudd_RecursiveDeref(ddman, dd1);
	Cudd_RecursiveDeref(ddman, dd2);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_Implies(DdManager *ddman, DdNode *dd1, DdNode *dd2)
{
	return DD_Or(ddman, DD_Not(ddman, dd1), dd2);
}

//------------------------------------------------------------------------------

DdNode *DD_Apply(DdManager *ddman, int op, DdNode *dd1, DdNode *dd2)
{
	DdNode *res;

	switch (op) {
		case APPLY_PLUS: res = Cudd_addApply(ddman, Cudd_addPlus, dd1, dd2); break;
		case APPLY_MINUS: res = Cudd_addApply(ddman, Cudd_addMinus, dd1, dd2); break;
		case APPLY_TIMES: res = Cudd_addApply(ddman, Cudd_addTimes, dd1, dd2); break;
		case APPLY_DIVIDE: res = Cudd_addApply(ddman, Cudd_addDivide, dd1, dd2); break;
		case APPLY_MIN: res = Cudd_addApply(ddman, Cudd_addMinimum, dd1, dd2); break;
		case APPLY_MAX: res = Cudd_addApply(ddman, Cudd_addMaximum, dd1, dd2); break;
		case APPLY_EQUALS: res = Cudd_addApply(ddman, Cudd_addEquals, dd1, dd2); break;
		case APPLY_NOTEQUALS: res = Cudd_addApply(ddman, Cudd_addNotEquals, dd1, dd2); break;
		case APPLY_GREATERTHAN: res = Cudd_addApply(ddman, Cudd_addGreaterThan, dd1, dd2); break;
		case APPLY_GREATERTHANEQUALS: res = Cudd_addApply(ddman, Cudd_addGreaterThanEquals, dd1, dd2); break;
		case APPLY_LESSTHAN: res = Cudd_addApply(ddman, Cudd_addLessThan, dd1, dd2); break;
		case APPLY_LESSTHANEQUALS: res = Cudd_addApply(ddman, Cudd_addLessThanEquals, dd1, dd2); break;
		case APPLY_POW: res = Cudd_addApply(ddman, Cudd_addPow, dd1, dd2); break;
		case APPLY_MOD: res = Cudd_addApply(ddman, Cudd_addMod, dd1, dd2); break;
		case APPLY_LOGXY: res = Cudd_addApply(ddman, Cudd_addLogXY, dd1, dd2); break;
		default: printf("\nError: Invalid APPLY operator.\n"); exit(1);
	}
	Cudd_Ref(res);
	Cudd_RecursiveDeref(ddman, dd1);
	Cudd_RecursiveDeref(ddman, dd2);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_MonadicApply(DdManager *ddman, int op, DdNode *dd)
{
	DdNode *res;

	switch (op) {
		case APPLY_FLOOR: res = Cudd_addMonadicApply(ddman, Cudd_addFloor, dd); break;
		case APPLY_CEIL: res = Cudd_addMonadicApply(ddman, Cudd_addCeil, dd); break;
		default: printf("\nError: Invalid monadic APPLY operator.\n"); exit(1);
	}
	Cudd_Ref(res);
	Cudd_RecursiveDeref(ddman, dd);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_Restrict(DdManager *ddman, DdNode *dd, DdNode *cube)
{
	DdNode *res;

	res = Cudd_addRestrict(ddman, dd, cube);
	Cudd_Ref(res);
	Cudd_RecursiveDeref(ddman, dd);
	Cudd_RecursiveDeref(ddman, cube);
	
	return res;
}

//------------------------------------------------------------------------------

DdNode *DD_ITE(DdManager *ddman, DdNode *dd1, DdNode *dd2, DdNode *dd3)
{
	DdNode *res;

	res = Cudd_addIte(ddman, dd1, dd2, dd3);
	Cudd_Ref(res);
	Cudd_RecursiveDeref(ddman, dd1);
	Cudd_RecursiveDeref(ddman, dd2);
	Cudd_RecursiveDeref(ddman, dd3);
	
	return res;
}

//------------------------------------------------------------------------------
