#include <stddef.h>
#include <iterator>

/* Implementation of single linked list */
namespace pk_code{


  template <class Tp>
  struct Single_List_node{
    Tp  L_data;
    Single_List_node* L_next;
  };

  template<typename Tp>
    struct Single_List_iterator
    {
      typedef Single_List_iterator<Tp>                Self;
      typedef Single_List_node<Tp>                    Node;

      typedef ptrdiff_t                         difference_type;
      typedef Tp                                value_type;
      typedef Tp*                               pointer;
      typedef Tp&                               reference;

      Single_List_iterator()
      : L_node() { }

      explicit
      Single_List_iterator(Node* p)
      : L_node(p) { }

      reference
      operator*() const
      { return (L_node)->L_data; }

      pointer
      operator->() const
      { return &((L_node)->L_data); }

      Self&
      operator++()
      {
	L_node = L_node->L_next;
	return *this;
      }

      Self
      operator++(int)
      {
	Self __tmp = *this;
	L_node = L_node->L_next;
	return __tmp;
      }

      bool
      operator==(const Self& x) const
      { return L_node == x.L_node; }

      bool
      operator!=(const Self& x) const
      { return L_node != x.L_node; }

      // The only member points to the %list element.
      Node* L_node;
    };
  /**
   *  @brief A list::const_iterator.
   *
   *  All the functions are op overloads.
  */
  template<typename Tp>
    struct Single_List_const_iterator
    {
      typedef Single_List_const_iterator<Tp>          Self;
      typedef const Single_List_node<Tp>              Node;
      typedef Single_List_iterator<Tp>                iterator;

      typedef ptrdiff_t                         difference_type;
      typedef Tp                                value_type;
      typedef const Tp*                         pointer;
      typedef const Tp&                         reference;

      Single_List_const_iterator()
      : L_node() { }

      explicit
      Single_List_const_iterator(const Node* x)
      : L_node(x) { }

      Single_List_const_iterator(const iterator& x)
      : L_node(x.L_node) { }

      reference
      operator*() const
      { return (L_node)->L_data; }

      pointer
      operator->() const
      { return &((L_node)->L_data); }

      Self&
      operator++()
      {
	L_node = L_node->L_next;
	return *this;
      }

      Self
      operator++(int)
      {
	Self tmp = *this;
	L_node = L_node->L_next;
	return tmp;
      }

      bool
      operator==(const Self& x) const
      { return L_node == x.L_node; }

      bool
      operator!=(const Self& x) const
      { return L_node != x.L_node; }

      // The only member points to the %list element.
      const Node* L_node;
    };

  template<typename Val>
    inline bool
    operator==(const Single_List_iterator<Val>& x,
	       const Single_List_const_iterator<Val>& y)
    { return x.L_node == y.L_node; }

  template<typename Val>
    inline bool
    operator!=(const Single_List_iterator<Val>& x,
               const Single_List_const_iterator<Val>& y)
    { return x.L_node != y.L_node; }

  template <class Tp, class Alloc>
  class Single_List
  {
  public:
    typedef Single_List_const_iterator<Tp>          Self;
    typedef Tp                                      value_type;
    typedef const Single_List_node<Tp>              Node;
    typedef Single_List_iterator<Tp>                iterator;
    typedef Single_List_const_iterator<Tp>          const_iterator;
    typedef Alloc                                   allocator_type;
    typedef size_t                                  size_type;
  protected:
    inline
    Node* get_node() { return allocator_type().allocate(1); }
    void del_node(Node *p) { allocator_type().deallocate(p, 1); }
  public:

    Single_List(const allocator_type&) {
      L_head = get_node();
      L_head->L_next = L_head;
    }
    ~Single_List() {
      clear();
      del_node(L_head);
    }
    void clear();
      /**
       *  Returns a read/write iterator that points to the first element in the
       *  %list.  Iteration is done in ordinary element order.
       */
    inline
    iterator begin()
      { return iterator(this->L_head.L_next); }

      /**
       *  Returns a read-only (constant) iterator that points to the
       *  first element in the %list.  Iteration is done in ordinary
       *  element order.
       */
    inline
    const_iterator begin() const
    { return const_iterator(this->L_head.L_next); }

      /**
       *  Returns a read/write iterator that points one past the last
       *  element in the %list.  Iteration is done in ordinary element
       *  order.
       */
    inline
    iterator end()
    { return iterator(&this->L_head); }

      /**
       *  Returns a read-only (constant) iterator that points one past
       *  the last element in the %list.  Iteration is done in ordinary
       *  element order.
       */
    inline
    const_iterator end() const
    { return iterator(&this->L_head); }

      /**
       *  Returns true if the %list is empty.  (Thus begin() would equal
       *  end().)
       */
    inline
    bool empty() const
    { return L_head.L_next == L_head; }

      /**  Returns the number of elements in the %list. Running time is linear
       * in the number of elements */
    inline
    size_type size() const
    { return std::distance(begin(), end()); }

    /**  Returns the size() of the largest possible %list.  */
    inline
    size_type max_size() const
    { return allocator_type().max_size(); }

      /**
       *  @brief  Add data to the front of the %list.
       *  @param  x  Data to be added.
       *
       *  This is a typical stack operation.  The function creates an
       *  element at the front of the %list and assigns the given data
       *  to it.  Due to the nature of a %list this operation can be
       *  done in constant time, and does not invalidate iterators and
       *  references.
       */
    inline
    void push_front(const value_type& x)
    { insert_after(L_head, x); }
      /**
       *  @brief  Removes first element.
       *
       *  This is a typical stack operation.  It shrinks the %list by
       *  one.  Due to the nature of a %list this operation can be done
       *  in constant time, and only invalidates iterators/references to
       *  the element being removed.
       *
       *  Note that no data is returned, and if the first element's data
       *  is needed, it should be retrieved before pop_front() is
       *  called.
       */
    inline
    void pop_front()
    { erase_after(L_head); }
      /**
       *  @brief  Remove element after given position.
       *  @param  position  Iterator pointing before element to be erased.
       *  @return  An iterator pointing to the next element (or end()).
       *
       *  This function will erase the element after the given position and thus
       *  shorten the %list by one.
       *
       *  Due to the nature of a %list this operation can be done in
       *  constant time, and only invalidates iterators/references to
       *  the element being removed.  The user is also cautioned that
       *  this function only erases the element, and that if the element
       *  is itself a pointer, the pointed-to memory is not touched in
       *  any way.  Managing the pointer is the user's responsibility.
       */
    inline
    iterator erase_after(iterator __position){
    }

  protected:
    Node* L_head;
  };

  template <class Tp, class Alloc>
  void
  Single_List<Tp,Alloc>::clear()
  {
    typedef Single_List_node<Tp> Node;
    Node* cur = L_head.L_next;
    while (cur != L_head) {
      Node* tmp = cur;
      cur = (cur->L_next);
      allocator_type().destroy(&tmp->L_data);
      put_node(tmp);
    }
    L_head->L_next = L_head;
  }
}
