# -*- coding: utf-8 -*-

"""
This module implements a dialog that allows users to derive attributes and encodings
"""
from PyQt4 import QtGui, QtCore
from PyQt4.QtGui import QDialog, QInputDialog, QLineEdit, QMessageBox
from PyQt4.QtCore import pyqtSignature

from Ui_derivedattribute import Ui_DerivedAttribute

import bitstring

class DerivedAttribute(QDialog, Ui_DerivedAttribute):
    def __init__(self, values,  column, dir, parent = None):
        """
        Constructor
        """
        QDialog.__init__(self, parent)
        self.setupUi(self)
        self.radCopy.setChecked(True)
        
        self.cmbType.addItem('Copy')
        self.cmbType.addItem('Reverse')
        self.cmbType.addItem('ASCII')
        self.cmbType.addItem('Integer')
        
        self.init_table(self.tblFiles, values, column)
        self.working_dir = dir
        
        # connect double click event of table header to custom procedure
        self.connect(self.tblFiles.horizontalHeader(), QtCore.SIGNAL("sectionDoubleClicked(int)"), self.modify_header)
        
    def modify_header(self, column):
        # Spawn dialog to edit headeritem
        text = self.tblFiles.horizontalHeaderItem(column).text()
        header, ok = QInputDialog.getText(self, "Edit header name", "Header name",  QLineEdit.Normal, text)
        if ok:
            a = QtGui.QTableWidgetItem(str(header))
            a = self.set_bold(a)
            self.tblFiles.setHorizontalHeaderItem(column, a)
        
    def set_bold(self, a):
        f = a.font()
        f.setBold(True)
        a.setFont(f)
        return a
        
    def init_table(self, table, values, column):
        # Column 1
        table.setColumnCount(1)
        table.setRowCount(len(values)-1)

        # create header items
        a = QtGui.QTableWidgetItem("bitstring")
        a = self.set_bold(a)
        table.setHorizontalHeaderItem(0, a)
        
        for t in range(1, len(values)):
            a = QtGui.QTableWidgetItem(values[t][0])
            table.setItem(t-1, 0, a)

        # Column 2
        if column > 0:
            # create header item
            table.setColumnCount(2)
            a = QtGui.QTableWidgetItem(values[0][column])
            a = self.set_bold(a)
            table.setHorizontalHeaderItem(1, a)
        
            for t in range(1, len(values)):
                a = QtGui.QTableWidgetItem(values[t][column])
                table.setItem(t-1, 1, a)
            
    def derive_attribute(self, table, column, method, pars = []):
        values = []
        if method == "Copy":
            # Create a copy of the attribute
            attr_name = table.horizontalHeaderItem(column).text()
            for r in range(table.rowCount()):
                values.append(table.item(r, column).text())
        elif method == "Cyclic":
            # Create a cyclic attribute with given index and length
            index = pars[0]            
            length = pars[1]
            if index >= length:
                QMessageBox.warning(self, "Cycle error", "Index must be between 0 and Length")
                return
                
            # Create header
            attr_name = table.horizontalHeaderItem(column).text()
            attr_name += " (" +  str(index) + "/" + str(length) + ")"
            
            # Copy attribute values
            for r in range(table.rowCount()):
                v = length*((r-index)/length)+index
                if v < 0:
                    values.append("")
                else:
                    values.append(table.item(v, column).text())
                
        elif method == "Encoding":
            # parse parameters
            fr = pars[0]
            to = pars[1]
            type = pars[2]
            
            if type == 'Copy':
                # Create header
                attr_name = "Copy " + str(fr) + "-" + str(to)
                
                # Derive reversed attribute values
                for r in range(table.rowCount()):
                    file = self.working_dir + table.item(r, 0).text()
                    b = bitstring.BitString(filename=file).bin[2+fr:2+to]
                    values.append(b[:])
            elif type == 'Reverse':
                # Create header
                attr_name = "Rev " + str(fr) + "-" + str(to)
                
                # Derive reversed attribute values
                for r in range(table.rowCount()):
                    file = self.working_dir + table.item(r, 0).text()
                    b = bitstring.BitString(filename=file).bin[2+fr:2+to]
                    values.append(b[::-1])
            elif type == "ASCII":
                # Verify that conversion to ASCII is possible
                if (to - fr) % 8 <> 0:
                    QMessageBox.warning(self, "ASCII error", "Please select a range whose length is a multiple of 8")
                    return
                else:
                    # Create header
                    attr_name = "ASCII " + str(fr) + "-" + str(to)
                    
                    # Derive ASCII value
                    for r in range(table.rowCount()):
                        file = self.working_dir + table.item(r, 0).text()
                        b = bitstring.BitString(filename=file).bin[2+fr:2+to]
                        value = "".join([chr(int(b[i:i+7],2)) for i in range(len(b)/8)])
                        values.append(value)
            elif type == "Integer":
                # Verify that conversion to int is possible
                if to - fr > 32:
                    QMessageBox.warning(self, "Integer error", "Please select a range with length at most 32")
                    return
                else:
                    # Create header
                    attr_name = "Int " + str(fr) + "-" + str(to)
                    
                    # Derive Integer value
                    for r in range(table.rowCount()):
                        file = self.working_dir + table.item(r, 0).text()
                        b = bitstring.BitString(filename=file).bin[2+fr:2+to]
                        value = str(int(b, 2))
                        values.append(value)
        
        # Add attribute to table
        if values <> []:
            self.add_attribute(table, attr_name,  values)
                
    def add_attribute(self, table, attr_name, values):
        c = table.columnCount()
        table.setColumnCount(c+1)
        
        # create header item
        a = QtGui.QTableWidgetItem(attr_name)
        a = self.set_bold(a)
        table.setHorizontalHeaderItem(c, a)
        
        # create values
        for v in range(0, len(values)):
            a = QtGui.QTableWidgetItem(values[v])
            table.setItem(v, c, a)            
    
    @pyqtSignature("bool")
    def on_radCyclic_toggled(self, checked):
        self.edtLength.setEnabled(checked)
        self.edtIndex.setEnabled(checked)
        self.lblLength.setEnabled(checked)
        self.lblIndex.setEnabled(checked)
    
    @pyqtSignature("bool")
    def on_radEncoding_toggled(self, checked):
        self.edtFrom.setEnabled(checked)
        self.edtTo.setEnabled(checked)
        self.cmbType.setEnabled(checked)
        self.lblFrom.setEnabled(checked)
        self.lblTo.setEnabled(checked)
        self.lblType.setEnabled(checked)
    
    @pyqtSignature("")
    def on_btnDerive_clicked(self):
        if self.radCopy.isChecked() == True:
            self.derive_attribute(self.tblFiles, 1, "Copy")
        elif self.radCyclic.isChecked() == True:
            pars = [int(self.edtIndex.text()),  int(self.edtLength.text())]
            self.derive_attribute(self.tblFiles, 1, "Cyclic", pars)
        elif self.radEncoding.isChecked() == True:
            pars = [int(self.edtFrom.text()), int(self.edtTo.text()), str(self.cmbType.currentText())]
            self.derive_attribute(self.tblFiles, 1, "Encoding", pars)
        
