# -*- coding: utf-8 -*-

"""
The main window of mCarve
"""

import pickle
import bitstring
import time
import hashlib
import Image
import os
import platform
from PyQt4 import QtGui,  QtCore
from PyQt4.QtGui import QMainWindow,  QFileDialog, QInputDialog, QLineEdit, QColor, QMessageBox
from PyQt4.QtCore import pyqtSignature,  QDir,  QString,  QSettings, QVariant, QSignalMapper, Qt,  QT_VERSION_STR, PYQT_VERSION_STR

from Ui_mainwindow import Ui_MainWindow
from derivedattribute import DerivedAttribute
from preferences import dlgPreferences
import carving
import layout

__version__ = "1.0.0"
__date__ = "June 7, 2011"

class MainWindow(QMainWindow, Ui_MainWindow):
    """
    Class documentation goes here.
    """
    def __init__(self, parent = None):
        """
        Constructor
        """
        # Global variables
        self.filename = None
        self.current_file = ''
        self.project_file = ''
        self.working_dir = str(QDir.currentPath()).split('/ui')[0]
        self.mask = bitstring.BitString('0x0')
        self.differences = []
        self.column = 0
        self.row = 0
        self.dump_len = 0
        self.display_from = 0
        self.display_to = 0
        self.span = []
        self.show_all = True
        self.selected_columns = []
        
        # General GUI settings
        QMainWindow.__init__(self, parent)
        self.setupUi(self)
        self.init_table(self.tblFiles, 1)
        self.init_table(self.tblFilesDerived)
        self.activeTable = self.tblFiles
        self.txtFile.setFontFamily('courier')
        self.txtFile.setFontPointSize(12)
        self.pbProgress.setValue(0)
        
        # Context menu of txtFile
        self.txtFile.setContextMenuPolicy(QtCore.Qt.ActionsContextMenu)
        self.actionEncoding.setEnabled(False)
        self.txtFile.addAction(self.actionEncoding)
        self.actionDeriveFile.setEnabled(False)
        self.txtFile.addAction(self.actionDeriveFile)
        self.actionDeriveProject.setEnabled(False)
        self.txtFile.addAction(self.actionDeriveProject)
        
        # connect double click event of table header to custom procedure
        # note: there does not seem to be a way to pass a second argument to a slot (not even with QSignalMapper)
        # A better solution would be to subclass the QTableWidget (or QHeaderItem)
        self.connect(self.tblFiles.horizontalHeader(), QtCore.SIGNAL("sectionDoubleClicked(int)"), self.modify_header)
        self.connect(self.tblFilesDerived.horizontalHeader(), QtCore.SIGNAL("sectionDoubleClicked(int)"), self.modify_header_derived)
        
        # Restore geometry, splitter sizes, constants, etc.
        settings = QSettings()
        self.restoreGeometry(settings.value("Geometry").toByteArray())
        self.columns = settings.value("columns", 60).toInt()[0]
        self.selected_only = settings.value("selected_only", QVariant(True)).toBool()
        self.ignore_empty = settings.value("ignore_empty", QVariant(True)).toBool()
        colors = settings.value("color_list", QVariant([[0, 255, 0], [255, 0, 0], [255, 255, 0]])).toList()
        self.colors = [tuple([item.toInt()[0] for item in c.toList()]) for c in colors]
        self.display_from = settings.value("display_from", 0).toInt()[0]
        self.display_to = settings.value("display_to", 8192).toInt()[0]

        widgets = self.findChildren(QtGui.QSplitter)
        for i, w in enumerate(widgets):
            w.restoreState(settings.value("splitter" + str(i)).toByteArray())
        
    def closeEvent(self, event):
        # Store geometry, splitter size, constants, etc.
        settings = QSettings()
        settings.setValue("Geometry", QVariant(self.saveGeometry()))
        widgets = self.findChildren(QtGui.QSplitter)
        for i, w in enumerate(widgets):
            settings.setValue("splitter" + str(i), QVariant(w.saveState()))
            
        settings.setValue("columns", self.columns)
        color_list = [list(c) for c in self.colors]
        settings.setValue("color_list", QVariant(color_list))
        settings.setValue("selected_only", self.selected_only)
        settings.setValue("ignore_empty", self.ignore_empty)
        settings.setValue("display_from", self.display_from)
        settings.setValue("display_to", self.display_to)
    
    def log(self, m):
        self.txtLog.append(m)
        
    def modify_header(self, column):
        # Spawn dialog to edit headeritem of tblFiles
        text = self.tblFiles.horizontalHeaderItem(column).text()
        header, ok = QInputDialog.getText(self, "Edit header name", "Header name",  QLineEdit.Normal, text)
        if ok:
            a = QtGui.QTableWidgetItem(str(header))
            a = self.set_bold(a)
            self.tblFiles.setHorizontalHeaderItem(column, a)
            
    def modify_header_derived(self, column):
        # Spawn dialog to edit headeritem of tblFilesDerived
        text = self.tblFilesDerived.horizontalHeaderItem(column).text()
        header, ok = QInputDialog.getText(self, "Edit header name", "Header name",  QLineEdit.Normal, text)
        if ok:
            a = QtGui.QTableWidgetItem(str(header))
            a = self.set_bold(a)
            self.tblFilesDerived.setHorizontalHeaderItem(column, a)
    
    @pyqtSignature("")
    def init_table(self, table, nr_of_attr = 0):
        # Set up table
        table.setColumnCount(nr_of_attr + 1)

        # Create header items
        a = QtGui.QTableWidgetItem("bitstring")
        a = self.set_bold(a)
        table.setHorizontalHeaderItem(0, a)

        # Fill up table
        for i in range(nr_of_attr):
            a = QtGui.QTableWidgetItem('attr ' + str(i+1))
            a = self.set_bold(a)
            table.setHorizontalHeaderItem(i+1, a)
        
    def fill_table(self, table, values):
        # Fill table with values
        table.setRowCount(len(values)-1)
        table.setColumnCount(len(values[0]))
        
        # Create headers
        for j in range(len(values[0])):
            a = QtGui.QTableWidgetItem(values[0][j])
            a = self.set_bold(a)
            table.setHorizontalHeaderItem(j, a)
            
        # Fill up table
        for i in range(1, len(values)):
            for j in range(len(values[i])):
                a = QtGui.QTableWidgetItem(values[i][j])
                table.setItem(i-1, j, a)
        
    def working_dir(self):
        raise Exception, "Called a deleted method!"
        return str(QDir.currentPath()).split('/ui')[0]
    
    def set_bold(self, a):
        f = a.font()
        f.setBold(True)
        a.setFont(f)
        return a
            
    @pyqtSignature("")
    def on_actionNew_triggered(self):
        """
        This method handles creation of new projects. The user is asked for a project file name and tables are initialized.
        """
        filename = QFileDialog.getSaveFileName (None, "", self.working_dir, "Project files (*.prj);;All files (*.*)")
        if filename:
            # Store filename and base directory
            filename = str(filename)
            self.working_dir = os.path.dirname(filename)
            self.project_file = filename
            
            # Start with empty file/attribute table
            self.init_table(self.tblFiles, 1)
            self.init_table(self.tblFilesDerived)
            
            # Save initial project
            self.save_project(self.project_file)
    
    @pyqtSignature("")
    def on_actionOpen_triggered(self):
        """
        This method handles opening project files. Some of the global variables are initialized.
        """
        filename = QFileDialog.getOpenFileName(None, "", self.working_dir, "Project files (*.prj);;All files (*.*)")
        if filename:
            filename = str(filename)
            input = open(filename, 'rb')
            self.project_file = filename
            self.working_dir = os.path.dirname(filename)
            
            table = pickle.load(input)
            try:
                table_derived = pickle.load(input)
            except EOFError:
                # Must have been an old file without derived attributes
                table_derived = [[t[0]] for t in table]
            
            self.fill_table(self.tblFiles, table)
            self.fill_table(self.tblFilesDerived, table_derived)
            input.close()
            # set focus on first dump and update display length
            if self.tblFiles.rowCount() > 0:
                f = self.tblFiles.item(0, 0).text()
                self.row = 0
                self.column = 0
                self.filename = self.working_dir + f
                self.dump_len = len(bitstring.BitString(filename=self.filename))
                self.display_to = self.dump_len
                self.display_file()
        
    def get_table(self, table):
        # Convert a GUI table to a table that can be pickled.
        c = table.columnCount()
        r = table.rowCount()
        
        tbl = []
        row = []
        for j in range(c):
            try:
                t = table.horizontalHeaderItem(j).text()
                t = str(t)
            except AttributeError:
                t = ''
            row.append(t)
        tbl.append(row)

        for i in range(r):
            row = []
            for j in range(c):
                try:
                    t = table.item(i, j).text()
                    t = str(t)
                except AttributeError:
                    t = ''
                row.append(t)
            tbl.append(row)
        
        return tbl
        
    def save_project(self, filename):
        # Pickle project to a file
        
        # Store attribute table in temporary data structure
        table = self.get_table(self.tblFiles)
        table_derived = self.get_table(self.tblFilesDerived)
        
        # Try to store the project
        try:
            output = open(filename, 'wb')
        except:
            pass
        else:
            pickle.dump(table, output)
            pickle.dump(table_derived, output)
            output.close()
    
    @pyqtSignature("")
    def on_actionSave_triggered(self):
        """
        This method deals with storing projects.
        """
        self.save_project(self.project_file)
        
    @pyqtSignature("")
    def on_actionSaveAs_triggered(self):
        """
        This method deals with storing projects.
        """
        filename = QFileDialog.getSaveFileName (None, "", self.working_dir, "Project files (*.prj);;All files (*.*)")
        if filename:
            # Store filename and base directory
            filename = str(filename)
            dirname = os.path.dirname(filename)
            
            if dirname <> self.working_dir:
                self.log("Save failed: project cannot be saved in a different folder.")
            else:
                self.working_dir = dirname
                self.project_file = filename
                self.save_project(self.project_file)
    
    @pyqtSignature("")
    def on_actionExit_triggered(self):
        """
        This method deals with the exit action.
        """
        self.close()
    
    @pyqtSignature("")
    def on_actionAdd_Attribute_triggered(self):
        """
        Add a user-defined attribute to the upper table.
        """
        c = self.tblFiles.columnCount()
        self.tblFiles.setColumnCount(c+1)
        a = QtGui.QTableWidgetItem("attr")
        a = self.set_bold(a)
        self.tblFiles.setHorizontalHeaderItem(c,a)

    @pyqtSignature("")
    def on_actionAdd_md5_Attribute_triggered(self):
        """
        Adds an MD5 attribute to the upper table. The value of the MD5 attribute is determined 
        by taking the MD5 hash of the dump. Therefore, the value of the attribute is the same if 
        and only if the dumps are the same.
        """
        c = self.tblFiles.columnCount()
        self.tblFiles.setColumnCount(c+1)
        a = QtGui.QTableWidgetItem("md5attr")
        a = self.set_bold(a)
        self.tblFiles.setHorizontalHeaderItem(c, a)

        for i in range(0, self.tblFiles.rowCount()):
            try:
                f = open(self.working_dir + self.tblFiles.item(i, 0).text(), 'r')
            except:
                pass
            else:
                s = f.read()
                m = hashlib.md5()
                m.update(s)
                md5value = m.hexdigest()
                a = QtGui.QTableWidgetItem(md5value)
                self.tblFiles.setItem(i, c, a)

    @pyqtSignature("")
    def on_actionAscending_triggered(self):
        """
        Sort the selected column A->Z.
        """
        selected = self.activeTable.selectedIndexes()
        selected = [x.column() for x in selected]
        self.activeTable.sortItems(selected[0],0)

    @pyqtSignature("")
    def on_actionDescending_triggered(self):
        """
        Sort the selected column Z->A.
        """
        selected = self.activeTable.selectedIndexes()
        selected = [x.column() for x in selected]
        self.activeTable.sortItems(selected[0],1)

    @pyqtSignature("")
    def on_actionAdd_Files_triggered(self):
        """
        This method lets the user select multiple files to be added to the current project.
        """
        if self.project_file == '':
            self.log('Please create a project first.')
            return
            
        filenames = QFileDialog.getOpenFileNames(self, QString(), self.working_dir)
        for i, f in enumerate(filenames):
            f = f.split(self.working_dir)
            if len(f) > 1:
                r = self.tblFiles.rowCount()
                self.tblFiles.setRowCount(r+1)
                a = QtGui.QTableWidgetItem(f[1])
                self.tblFiles.setItem(r, 0, a)
            else:
                self.log("Failed to add " + str(f[0]) + ": can only add files that are in the subtree of the project file.")    
        
    def display_file(self):
        # Remember scrollbar position and clear textedit
        sb = self.txtFile.verticalScrollBar().value()
        self.txtFile.clear()
        
        if self.filename == None:
            return
        
        file = self.filename
        
        # Read the currently selected bitstring
        ub = int(self.display_to)
        dump = bitstring.BitString(filename=file).bin[2:ub+2]
        
        # Enable deriving files/projects and encodings
        self.actionEncoding.setEnabled(True)
        self.actionDeriveFile.setEnabled(True)
        self.actionDeriveProject.setEnabled(True)
            
        # First create spans for commons bits
        if self.mask == bitstring.BitString('0x0'):
            self.span = ['w']*(len(dump))
        else:    
            self.span = layout.bin_to_span(self.mask)
            
        # Add differences to the spans
        if self.differences != []:
            if self.show_all == True:
                # Show all
                for s in self.differences:
                    self.span = layout.add_differences(self.span, s[0], s[0]+s[1])
            else:
                # Show one
                r = self.currentDifferences
                min = self.differences[r][0]
                max = self.differences[r][0]+self.differences[r][1]
                self.span = layout.add_differences(self.span, min, max)
        l = layout.add_span_to_bin(layout.bin_to_list(dump), self.span)
             
        # Add line breaks according to user-supplied value for number of columns
        l = layout.list_add_linebreaks(l, self.columns)
        
        # Create HTML
        self.txtFile.setHtml(layout.html(l, self.colors))
        
        # Restore scrollbar position
        self.txtFile.verticalScrollBar().setValue(sb)

    @pyqtSignature("int, int, int, int")
    def on_tblFiles_currentCellChanged(self, currentRow, currentColumn, previousRow, previousColumn):
        """
        If the user changes the active row in the upper table, the textedit is updated with 
        the dump of the current row
        """
        if (currentRow <> previousRow):
            f = self.tblFiles.item(currentRow, 0).text()
            self.row = currentRow
            self.column = currentColumn
            self.filename = self.working_dir + f
            self.display_file()
    
    def bundles(self, c, selected=[]):
        """
        This method transforms a table into a set of bundles
        """
        # bounds
        lb = self.display_from
        ub = self.display_to
        
        # If selected only is chosen, only selected rows must be included.
        if selected == []:
            if self.selected_only == True:
                selected = self.activeTable.selectedIndexes()
                selected = [x.row() for x in selected]
            else:
                selected = range(self.activeTable.rowCount()) 
        
        # Create dictionary structure for all files
        d = {}
        for i in selected:
            file = str(self.activeTable.item(i, 0).text())
            value = str(self.activeTable.item(i, c).text())
            if value <> '' or self.ignore_empty == False:
                try:
                    d[value].append(file)
                except KeyError:
                    d[value] = []
                    d[value].append(file)
                
        # Add pointers to dumps to the bundles
        bundles = []
        for i in d:
            bundle = []             
            for j in d[i]:
                el = bitstring.BitString(filename=self.working_dir + j)[lb:ub]
                bundle.append(el)            
            bundles.append(bundle)
        return bundles
        
    @pyqtSignature("")
    def on_btnCommon_clicked(self):
        """
        This method calls the common procedure if the user triggered the action.
        """
        # Determine which column is currently active
        c = self.activeTable.currentColumn()
        b = self.bundles(c)
        
        # Verify whether sufficient dumps/bundles are selected and call the common procedure.
        if (len(b) > 1 or len(b[0]) > 1):
            I = carving.ones(len(b[0][0]))
            I = carving.common_bits_bundle(b, I)
            self.mask = I
            self.display_file()
        else:
            self.log("Please select more than one bitstring.")
    
    @pyqtSignature("")
    def on_btnDifferent_clicked(self):
        """
        This method calls the different procedure if the user triggered the action.
        """
        # Reset current differences.
        self.differences = []
        
        # Determine which column is currently active
        c = self.activeTable.currentColumn()
        bb = self.bundles(c)
        
        # Select one candidate from every bundle
        sb = []
        for b in bb:
            sb.append(b[0])
        
        # Call different procedure if dumps of more than one bundle are selected
        if len(sb) > 1:
            I = carving.different_minimal(sb, self.pbProgress)
            self.differences = I
            self.currentDifferences = 0
            self.show_all = True
        else:
            self.txtLog.setText("Please select bitstrings of more than one bundle")
            
        # Display differences in list
        self.display_differences()
  
    def display_differences(self):
        self.lstDifferences.clear()
        for d in self.differences:
            self.lstDifferences.addItem(str(d[0]) + '-' + str(d[0]+d[1]-1))
            
    @pyqtSignature("")
    def on_btnCombined_clicked(self):
        """
        This method combines the common and dissim procedures. It first calls the common 
        procedure and uses the output as a mask to the differences procedure.
        """
        self.differences = []        
        c = self.activeTable.selectedItems()[0].column()
        bb = self.bundles(c)
                
        # Apply common
        I = carving.ones(len(bb[0][0]))
        I = carving.common_bits_bundle(bb, I)
        self.mask = I
        
        # Bundles of bitstrings really are bundles of pointers....
        # therefore, need to be recreated.
        bb = self.bundles(c)
                
        # Select one candidate from every bundle
        sb = []
        for b in bb:
            sb.append(b[0])
            
        # Apply different_mask
        if len(sb) > 1:
            I = carving.different_mask_minimal(sb, I, self.pbProgress)
            self.differences = I
            self.currentDifferences = 0
            self.show_all = True
            self.display_differences()
        else:
            self.txtLog.append("Nothing to carve: Less than two bundles.")
        
        # Display the results
        self.display_differences()
        
    @pyqtSignature("")
    def on_txtFile_selectionChanged(self):
        """
        This method keeps track of the current selection in the textedit.
        """
        if self.differences == []:
            return False
        
        t = self.txtFile.textCursor()
        i = int(t.position())
        cols = self.columns
                
        # For some reason whitespace at the end of the line counts as a character
        i = i-i/(cols+1)
        
        z = -1
        x = 0
        while (z == -1):
            if self.differences[x][0] + self.differences[x][1] < i:
                x +=1
            else:
                z = x
        self.lstDifferences.setCurrentRow(z)
        
    @pyqtSignature("int")
    def on_lstDifferences_currentRowChanged(self, currentRow):
        """
        This method updates the display if the selected difference interval is changed.
        """
        self.currentDifferences = currentRow
        self.show_all = False
        self.display_file()
    
    @pyqtSignature("")
    def on_btnShowAll_clicked(self):
        """
        This method updates the display and adds all differences to it.
        """
        self.show_all = True
        self.display_file()
    
    @pyqtSignature("")
    def on_actionLaTeX_triggered(self):
        """
        This method creates latex output of the textedit.
        """
        #obtain display values
        cols = self.columns
        ub = int(self.display_to)
        filename = QFileDialog.getSaveFileName (None, "", self.working_dir + "/commonfile.tex", "LaTeX file (*.tex)")
        
        thebits = layout.bin_to_list(bitstring.BitString(filename=self.filename).bin[2:ub+2])
        themask = self.span
        linelen = 128 # determines when lines are wrapped
        commonbits = "" # to build the file
        for i in range(0,len(thebits)):
            # wrap a line
            if i % linelen == 0 and i != 0:
                commonbits += r"\myendline" + "\n"
            if i % (4 * linelen) == 0:
                commonbits += r"\extraline" + "\n"
        
            # construct the bits
            if themask[i] == 'r':
                commonbits += r"\x"  # a red bit (ignore 0/1)
            else:
                if thebits[i] == 1:
                    commonbits += r"\n" # a 1 in green part
                else:
                    commonbits += r"\z" # a 0 in green part
        
        commonbits += r"\myendline" + "\n"
        
        # write to file
        file = open(filename, "w")
        file.write(commonbits)
        file.close()
    
    @pyqtSignature("")
    def on_actionAdd_const_Attribute_triggered(self):
        """
        Add a constant attribute with the same value for all dumps.
        """
        c = self.tblFiles.columnCount()
        self.tblFiles.setColumnCount(c+1)
        a = QtGui.QTableWidgetItem("constAttr")
        a = self.set_bold(a)
        self.tblFiles.setHorizontalHeaderItem(c, a)

        for i in range(0, self.tblFiles.rowCount()):
            a = QtGui.QTableWidgetItem(str(c))
            self.tblFiles.setItem(i, c, a)
    
    @pyqtSignature("")
    def on_actionJPEG_triggered(self):
        """
        This method creates a JPG image of the textedit.
        """
        def drawblock(image, s, x, y, color):
            for i in range(0, s):
                for j in range(0, s):
                    image.putpixel(((s*x)+i, (s*y)+j), color)
            
        #obtain display values
        cols = self.columns
        ub = int(self.display_to)
        
        filename = QFileDialog.getSaveFileName (None, "", self.working_dir, "JPG files(*.jpg)")
        thebits = layout.bin_to_list(bitstring.BitString(filename=self.filename).bin[2:ub+2])
        themask = self.span
        size = 10
        linelen = 64 # determines when lines are wrapped
        c = linelen
        r = len(themask)/c
        
        im = Image.new('RGB', (size*c, size*r), (0, 0, 0))
        
        for i in range(0, r):
            for j in range(0, c):
                # construct the bits
                p = (i*c)+j
                if themask[p] == 'r':
                    drawblock(im, size, j, i, self.colors[0])
                else:
                    if thebits[p] == 1:
                        drawblock(im, size, j, i, self.colors[1])
                    else:
                        drawblock(im, size, j, i, (0, 0, 255))

        f = open(str(filename), 'w')
        im.save(str(filename), "JPEG")
        f.close()
    
    @pyqtSignature("")
    def on_actionPerformance_triggered(self):
        """
        Slot documentation goes here.
        """
        # Select attribute
        c = self.tblFiles.currentColumn()
        
        self.log("======================================================================")
        self.log("= Performance test:")
        self.log("= This test provides timing information of the combined procedure.")
        self.log("= It runs the procedure on a dumpset of b bundles (for 0 \leq b < 10)")
        self.log("= Running on attribute " + str(c))    
        self.log("======================================================================")
        num_bundles = 0
        i = 0
        while (num_bundles < 10 and i < self.tblFiles.rowCount() - 1):
            i += 1
            # Execute code
            bb = self.bundles(c, range(0, i))
            
            # only execute code when there is a new bundle    
            if len(bb) > num_bundles:
                num_bundles = len(bb)
                # Start time measuring
                start = time.time()
                
                # Apply common
                I = carving.ones(len(bb[0][0]))
                I = carving.common_bits_bundle(bb, I)
                self.mask = I
                middle = time.time()
            
                # Bundles of bitstrings really are bundles of pointers....
                # therefore, need to be recreated.
                bb = self.bundles(c, range(0, i))
                
                # Select one candidate from every bundle
                sb = []
                for b in bb:
                    sb.append(b[0])
            
                # Apply different_mask
                if len(sb) > 1:
                    I = carving.different_mask_minimal(sb, I, self.pbProgress)
                    self.differences = I
                else:
                    self.log("Nothing to carve: Less than two bundles.")
        
                # Log
                stop = time.time()
                t = "%.3f" % (stop - start)
                self.log("(#bundles, #dumps) = (" + str(num_bundles) + "," + str(i) + 
                                    "); Time: " + t + " s." )
                
    @pyqtSignature("")
    def on_actionEncoding_triggered(self):
        """
        This method allows the user to apply an encoding to an interval. The encoded value is 
        added to the bottom table.
        """
        table = self.get_table(self.activeTable)
        dialog = DerivedAttribute(table, 0, self.working_dir)
        dialog.radEncoding.setChecked(True)
        cursor = self.txtFile.textCursor()
        
        # find fro and to        
        cols = self.columns
        fro = cursor.selectionStart() 
        fro = fro-fro/(cols+1)
        to = cursor.selectionEnd() 
        to = to-to/(cols+1)
        
        # populate dialog
        dialog.edtFrom.setText(str(fro))
        dialog.edtTo.setText(str(to))
        cursor = self.txtFile.textCursor()
        if dialog.exec_():
            columns = set([c.column() for c in dialog.tblFiles.selectedItems()])
            if columns == set([]):
                columns = set([1])
            
            for c in columns:
                attr_name = dialog.tblFiles.horizontalHeaderItem(c).text()
                values = []
                for r in range(dialog.tblFiles.rowCount()):
                    values.append(dialog.tblFiles.item(r, c).text())
                self.add_attribute(self.tblFilesDerived, attr_name, values)
                
    def add_attribute(self, table, attr_name, values):
        # This method adds an attribute to a table
        c = table.columnCount()
        table.setColumnCount(c+1)
        
        # create header item
        a = QtGui.QTableWidgetItem(attr_name)
        a = self.set_bold(a)
        table.setHorizontalHeaderItem(c, a)
        
        # create values
        for v in range(0, len(values)):
            a = QtGui.QTableWidgetItem(values[v])
            table.setItem(v, c, a)  
    
    @pyqtSignature("")
    def on_actionDerive_attribute_triggered(self):
        """
        This method allows the user to derive an attribute from another attribute. The 
        derived attribute is added to the bottom table.
        """
        c = self.tblFiles.selectedItems()
        if len(c) == 0 or c[0].column() == 0:
            self.log("Please select an attribute")
            return
        else:
            c = c[0].column()
        
        table = self.get_table(self.tblFiles)
        dialog = DerivedAttribute(table, c, self.working_dir)
        if dialog.exec_():
            try:
                columns = set([c.column() for c in dialog.tblFiles.selectedItems()])
            except IndexError:
                columns = set([1])
            for c in columns:
                attr_name = dialog.tblFiles.horizontalHeaderItem(c).text()
                # create list of derived attribute values (in right order)
                dict = {}
                values = []
                for r in range(dialog.tblFiles.rowCount()):
                    dict[str(dialog.tblFiles.item(r, 0).text())] = dialog.tblFiles.item(r, c).text()
                for r in range(self.activeTable.rowCount()):
                    values.append(dict[str(self.tblFilesDerived.item(r, 0).text())])
                
                self.add_attribute(self.tblFilesDerived, attr_name, values)
    
    @pyqtSignature("")
    def on_tblFiles_itemSelectionChanged(self):
        self.activeTable = self.tblFiles
        self.inactiveTable = self.tblFilesDerived
    
    @pyqtSignature("")
    def on_tblFilesDerived_itemSelectionChanged(self):
        self.activeTable = self.tblFilesDerived
        self.inactiveTable = self.tblFiles
    
    @pyqtSignature("")
    def on_actionPreferences_triggered(self):
        """
        This method calls the preferences dialog and records the preference settings as 
        supplied by the user.
        """
        dialog = dlgPreferences()
        dialog.set_colors([QColor(c[0], c[1], c[2], 255) for c in self.colors])
        dialog.chkSelectedOnly.setChecked(self.selected_only)
        dialog.chkIgnoreEmpty.setChecked(self.ignore_empty)
        dialog.edtColumns.setText(str(self.columns))
        dialog.edtFrom.setText(str(self.display_from))
        dialog.edtTo.setText(str(self.display_to))
        if dialog.exec_():
            colors = dialog.get_colors()
            for i in range(3):
                self.colors[i] = colors[i][0:3]
            self.selected_only = dialog.chkSelectedOnly.isChecked()
            self.ignore_empty = dialog.chkIgnoreEmpty.isChecked()
            self.columns = int(dialog.edtColumns.text())
            self.display_from = int(dialog.edtFrom.text())
            self.display_to = int(dialog.edtTo.text())
            self.display_file()
    
    def find_selection(self):
        # Find the currently selected area in the textedit.
        cursor = self.txtFile.textCursor()
        cols = self.columns
        fro = cursor.selectionStart() 
        fro = fro-fro/(cols+1)
        to = cursor.selectionEnd() 
        to = to-to/(cols+1)
        
        return fro,  to
    
    @pyqtSignature("")
    def on_actionDeriveFile_triggered(self):
        """
        This method allows the user to derive a file from a selection in the textedit.
        """
        # find fro and to        
        fro, to = self.find_selection()
        
        # populate dialog
        filename = QFileDialog.getSaveFileName (None, "", self.working_dir)
        if filename:
            # Store bits in filename
            filename = str(filename)
            file = open(filename, 'wb')
            
            dump = bitstring.BitString(filename=self.filename)[fro:to]
            dump.tofile(file)
            file.close()
            self.log('Successfully stored file: ' + filename)
        
    @pyqtSignature("")
    def on_actionDeriveProject_triggered(self):
        """
        This method allows the user to derive a project from the interval in the textedit. A copy 
        of the project file is created and dumps are derived by selecting the interval of the dumps 
        of the current project.
        """
        fro, to = self.find_selection()
              
        filename = QFileDialog.getSaveFileName (None, "", self.working_dir, "Project files (*.prj);;All files (*.*)")
        if filename:
            # Store filename and base directory
            filename = str(filename)
            root, ext = os.path.splitext(filename)
            project_name = os.path.splitext(os.path.split(filename)[1])[0]
            
            table = self.get_table(self.tblFiles)
            table_derived = self.get_table(self.tblFilesDerived)
            
            try:
                os.makedirs(root)
            except OSError:
                pass
            
            for i in range(self.tblFiles.rowCount()):
                file = str(self.tblFiles.item(i, 0).text())
                head, tail = os.path.split(file)
                
                dump_name = os.path.split(table[i+1][0])[1]
                table[i+1][0] =  "/" + project_name + "/" + dump_name
                table_derived[i+1][0] = "/" + project_name + "/" + os.path.split(table_derived[i+1][0])[1]
                
                # try to store file
                dump_file = root + "/" + dump_name
                handle = open(dump_file, 'wb')
                orig_dump = self.working_dir + file
                dump = bitstring.BitString(filename=orig_dump)[fro:to]
                dump.tofile(handle)
                handle.close()
             
            # Try to store the project
            try:
                output = open(filename, 'wb')
            except:
                pass
            else:
                pickle.dump(table, output)
                pickle.dump(table_derived, output)
                output.close()   
            
            self.log('Successfully stored project: ' + project_name)
    
    @pyqtSignature("")
    def on_actionConvergence_triggered(self):
        self.log("==============================================")
        self.log("= Convergence test:")
        self.log("= This test selects a set of dumps and verifies whether all intervals")
        self.log("= found by the combined procedure are within the attribute mapping ")
        self.log("= of the encoding of the attribute. If yes, it terminates, otherwise ")
        self.log("= it increases the set of dumps with one.")
        self.log("==============================================")
        max_num_attributes = 10
        
        # Select attribute
        for c in range(1, min(self.tblFiles.columnCount(), max_num_attributes)):
            # Unselect all cells except for the first one
            for i in range(0, self.tblFiles.rowCount()):
                for j in range(0, self.tblFiles.columnCount()):
                    self.tblFiles.item(i, j).setSelected(False)
            self.tblFiles.item(0, c).setSelected(True)
            s = str(self.tblFiles.horizontalHeaderItem(c).text())
            
            try:
                lower = int(s.split()[1].split("+")[0])
                upper = int(s.split()[1].split("+")[1]) + lower
            except:
                self.log("Invalid test set.")
                break

            for i in range(1, self.tblFiles.rowCount()):
                # Select one more item
                self.tblFiles.item(i, c).setSelected(True)
                       
                # Execute code
                self.on_btnCombined_clicked()
         
                # number of intervals            
                n = len(self.differences)
                
                # Log
                self.log("Attribute: " + str(c) + "; #dumps: " + str(i) + "; #intervals: " + str(n))
    
                low = self.differences[0][0]
                high = self.differences[n-1][0] + self.differences[n-1][1]
                
                if ((low >= lower) & (high <= upper)):
                    self.log("Attribute: " + str(c) + "; Convergence at #dumps = " + str(i))
                    break
    
    @pyqtSignature("")
    def on_actionDelete_File_triggered(self):
        """
        Delete row from both tables.
        """
        selected_items = self.tblFiles.selectedItems()
        sels = [s.row() for s in selected_items]
        
        # Figure out which rows must be removed from the bottom table
        d = []
        dumps = [self.tblFiles.item(s.row(), 0).text() for s in selected_items]
        for r in range(self.tblFilesDerived.rowCount()):
            if self.tblFilesDerived.item(r, 0).text() in dumps:
                d.append(r)
        
        # remove rows from top table
        for r in sorted(sels, reverse=True):
            self.tblFiles.removeRow(r)
        
        # remove rows from bottom table
        for r in sorted(d, reverse=True):
            self.tblFilesDerived.removeRow(r)
    
    @pyqtSignature("")
    def on_actionDelete_Attribute_triggered(self):
        """
        Delete the currently selected attribute from its table.
        """
        selected_items = self.activeTable.selectedItems()
        cols = set([s.column() for s in selected_items]).difference(set([0]))
        
        # remove rows from the active table
        for r in sorted(cols, reverse=True):
            self.activeTable.removeColumn(r)
    
    @pyqtSignature("")
    def on_action_About_triggered(self):
        """
        About window
        """
        QMessageBox.about(self, "About mCarve","""
            <b>mCarve</b> v %s, (%s)
            <p> Ton van Deursen, Sjouke Mauw, and Sa&#353;a Radomirovi&#263;.</p>
            <p> University of Luxembourg.</p>
            <p>The present project is supported by the National Research Fund, Luxembourg</p>
            <p><a href="http://satoss.uni.lu/mcarve">http://satoss.uni.lu/mcarve</a></p>
            <p><img src="%s/afr.jpg"></p>
            <p>Python %s - Qt %s - PyQt %s on %s</p>""" 
            % (__version__, __date__, os.path.dirname(__file__), platform.python_version(),QT_VERSION_STR, PYQT_VERSION_STR, platform.system()))
        
